/*
 * org.jdesktop.j3d.examples.distort_glyph.DistortBehavior.java / gold.jpg
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistribution of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistribution in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
 * USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
 * ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or
 * intended for use in the design, construction, operation or
 * maintenance of any nuclear facility.
 *
 * $Revision: 1.2 $
 * $Date: 2007/02/09 17:21:36 $
 * $State: Exp $
 */
package com.interactivemesh.j3d.testspace.jcanvas3d;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;

import java.awt.geom.GeneralPath;

import java.util.Enumeration;

import javax.media.j3d.Appearance;
import javax.media.j3d.Background;
import javax.media.j3d.Behavior;
import javax.media.j3d.BoundingBox;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.Geometry;
import javax.media.j3d.GeometryArray;
import javax.media.j3d.GeometryUpdater;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.Locale;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Texture;
import javax.media.j3d.TexCoordGeneration;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.TriangleArray;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.VirtualUniverse;
import javax.media.j3d.WakeupCriterion;
import javax.media.j3d.WakeupOnBehaviorPost;
import javax.media.j3d.WakeupOnElapsedFrames;
import javax.media.j3d.WakeupOr;

import javax.swing.JPanel;

import javax.vecmath.AxisAngle4d;
import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;
import javax.vecmath.Vector3f;
import javax.vecmath.Vector4f;

import com.sun.j3d.utils.image.TextureLoader;

// String3D 3.0, see: http://www.interactivemesh.org/testspace/awtshapeextruder.html
import com.interactivemesh.j3d.community.utils.geometry.AWTShapeExtruder;
import com.interactivemesh.j3d.community.utils.geometry.AWTShapeExtrusion;
import com.interactivemesh.j3d.community.utils.geometry.String3D;

// OrbitBehaviorInterim 2.1, see: http://www.interactivemesh.org/testspace/orbitbehavior.html
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim;

/**
 * DistortStringReflection.java
 *
 * Version: 2.0
 * Date: 2009/03/09
 * 
 * Author:
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * under consideration of the copyrigth notice above for the 
 * internal class 'DistortBehavior' and the image 'gold.jpg',
 * but you may not modify, compile, or distribute this 'DistortStringReflection.java'.
 *
 */
final class DistortStringReflection  {

    static {
        System.out.println("DistortStringReflection 2.0 : Copyright (c) 2009 August Lammersdorf, www.InteractiveMesh.com.");
    }
   
    private BoundingSphere      globalBounds        =   null;

    private View                view                =   null;
    private Locale              locale              =   null;
    private DistortStringReflectionJC3DOB jCanvas3D =   null;
    private JPanel              jPanel              =   null;

    private OrbitBehaviorInterim orbitBehInterim    =   null;

    private DistortBehavior     distortBeh          =   null;

    private BranchGroup         sceneBranch         =   null;
    private BranchGroup         viewBranch          =   null;
    private BranchGroup         enviBranch          =   null;

    private Color               bgColor             =   new Color(0.0f, 0.4f, 0.8f);
    
    private DistortStringReflectionPanel frame      =   null;

    DistortStringReflection(DistortStringReflectionPanel frame) {
        this.frame = frame;
        initUniverse();
    }
    
    //
    // Called by JCanvas3DOB
    //
    void updateFPS(int value) {
        frame.updateFPS(value);
    }

    //
    // Called by frame
    //
    
    JPanel getJCanvas3D() {
        return jPanel;
    }
    
    void closeUniverse() {
        view.removeAllCanvas3Ds();
        view.attachViewPlatform(null);
        locale.getVirtualUniverse().removeAllLocales();
    }

    void setVantagePoint(String vp) {
        Transform3D vpTransform = new Transform3D();
        if (vp.equalsIgnoreCase("Front")) {
            vpTransform.setTranslation(new Vector3d(0.0, 0.45, 3.4));
        }
        else if (vp.equalsIgnoreCase("Side")) {
            vpTransform.setRotation(new AxisAngle4d(0, 1, 0, Math.toRadians(-90)));
            vpTransform.setTranslation(new Vector3d(-3.4, 0.45, 0.0));
        }
        else
            return;
        orbitBehInterim.setViewingTransform(vpTransform, new Point3d());
    }
    
    // Animation
    
    void startStopDistort(boolean start) {
        if (start)
            distortBeh.postId(DistortBehavior.START);
        else
            distortBeh.postId(DistortBehavior.STOP);
    }
    
    // Reflection
    
    void setFraction(float fraction) {
        jCanvas3D.setFraction(fraction);
    }
    
    void setTopOpacity(float opacity) {
        jCanvas3D.setTopOpacity(opacity);
    }
    void setBottomOpacity(float opacity) {
        jCanvas3D.setBottomOpacity(opacity);
    }
    
    void setTopColorRGB(Color rgb) {
        jCanvas3D.setTopColorRGB(rgb);
    }
    void setBottomColorRGB(Color rgb) {
        jCanvas3D.setBottomColorRGB(rgb);
    }

    //
    // Init universe
    //
    private void initUniverse() {
        
        createUniverse();
        createScene();
        setLive();

        createJCanvas3D();

        distortBeh.postId(DistortBehavior.START);

        // Due to special treatment for JCanvas3D setup 'orbitBehInterim' at the end
        // Setup navigation

        double sceneRadius = 1.0f;

        Bounds bounds = sceneBranch.getBounds();
        BoundingSphere sphereBounds = null; 

        if (bounds.isEmpty()) {
            sphereBounds = new BoundingSphere(); 
        }
        else {
            if (sphereBounds instanceof BoundingSphere)
                sphereBounds = (BoundingSphere)bounds;
            else
                sphereBounds = new BoundingSphere(bounds);

        }
            
        sceneRadius = sphereBounds.getRadius();

        orbitBehInterim.setTransFactors(sceneRadius/4.0, sceneRadius/4.0);
        orbitBehInterim.setZoomFactor(sceneRadius/4.0);
        orbitBehInterim.setRotFactors(0.5, 0.5);
        
        orbitBehInterim.setClippingEnabled(true);
        orbitBehInterim.setClippingBounds(sphereBounds);

        orbitBehInterim.setProjectionMode(View.PARALLEL_PROJECTION);
        orbitBehInterim.setPureParallelEnabled(true);
        
        setVantagePoint("Front");

    }

    private void createScene() {

        // String3D Geometry
        
        GeneralPath extrPath = new GeneralPath();

        float extend = 3.0f;
        float depth = 50.0f;
        float cut = depth/10.0f;

        extrPath.moveTo(0.0f, 0.0f);
        extrPath.lineTo(cut, extend);
        extrPath.lineTo(depth-cut, extend);
        extrPath.lineTo(depth, 0.0f);

        AWTShapeExtrusion   extrusion   =   new AWTShapeExtrusion(extrPath);
        AWTShapeExtruder    extruder    =   new AWTShapeExtruder(0.25, extrusion, Math.toRadians(24));

        Font font = new Font("Dialog", Font.PLAIN, 100);

        String3D string3D = new String3D(font, extruder);
        string3D.setPosition(new Point3f(0, 0, 0));
        string3D.setCharacterSpacing(5.0f);
        string3D.setAlignment(String3D.Alignment.CENTER);
        string3D.setPath(String3D.Path.RIGHT);

        Transform3D scaleT3D = new Transform3D();
        scaleT3D.setScale(0.01);                            // scale down 1/100
        scaleT3D.setTranslation(new Vector3d(0, 0, -0.25));  // center z
        extruder.setGeometryTransform(scaleT3D);

        BoundingBox bBox = new BoundingBox();
        
        GeometryArray geom3D = string3D.getStringGeometry("123D", bBox);
        
        Point3d lower = new Point3d();
        Point3d upper = new Point3d();
        bBox.getLower(lower);
        bBox.getUpper(upper);

        // BY_REFERENCE needed
        int vertexCt = geom3D.getVertexCount();
        float[] coords = new float[vertexCt*3];
        float[] normals = new float[vertexCt*3];

        geom3D.getCoordinates(0, coords);
        geom3D.getNormals(0, normals);

//System.out.println("vertexCt = " + vertexCt);
//System.out.println("bBox = " + bBox);

        TriangleArray stringGeom = new TriangleArray(vertexCt, 
                                                     GeometryArray.COORDINATES | 
                                                     GeometryArray.NORMALS | 
                                                     GeometryArray.BY_REFERENCE);
        stringGeom.setCapability(GeometryArray.ALLOW_REF_DATA_WRITE);
        stringGeom.setCoordRefFloat(coords);
        stringGeom.setNormalRefFloat(normals);
        
        // Appearance
        
        Appearance appearGold = new Appearance();

        Texture texture = new TextureLoader(this.getClass().getResource("gold.jpg"), null).getTexture();
        texture.setMagFilter(Texture.NICEST);
        texture.setMinFilter(Texture.NICEST);

        appearGold.setTexture(texture);

        TexCoordGeneration texCoordGen = new TexCoordGeneration(TexCoordGeneration.SPHERE_MAP, TexCoordGeneration.TEXTURE_COORDINATE_2);
        
        double xScale = 1.0/(upper.x-lower.x);
        double yScale = 1.0f/(upper.y-lower.y) * 0.3f; // limit height of image
        float planeSx = (float)xScale;
        float planeSw = (float)(-lower.x * xScale);
        float planeTy = (float)yScale; 
        float planeTw = (float)(-lower.y * yScale);

//System.out.println("planeSx / planeSw / planeTy / planeTw= " + planeSx + " / " + planeSw + " / " + planeTy + " / " + planeTw);
        
        texCoordGen.setPlaneS(new Vector4f(planeSx, 0f, 0f, planeSw));
        texCoordGen.setPlaneT(new Vector4f(0.0f, planeTy, 0f, planeTw));

        appearGold.setTexCoordGeneration(texCoordGen);

        // Shape 3D

        Shape3D string3DShape = new Shape3D();
        string3DShape.setAppearance(appearGold);
        string3DShape.setGeometry(stringGeom);

        // DistortBehavior
        distortBeh = new DistortBehavior(stringGeom, 1000);
        distortBeh.setSchedulingBounds(globalBounds); // V 1.2
        distortBeh.setEnable(true);

        // Rotate string due to DistortBehavior's calculations
        TransformGroup rotateTG  = new TransformGroup();
        Transform3D rotateT3D = new Transform3D();
        rotateT3D.setEuler(new Vector3d(0, -Math.toRadians(80), 0));
        rotateTG.setTransform(rotateT3D);

        rotateTG.addChild(string3DShape);
        
        sceneBranch.addChild(rotateTG);
        sceneBranch.addChild(distortBeh);
    }

    // Set live
    private void setLive() {
        sceneBranch.compile();
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
    }

    // Base world
    private void createUniverse() {
        // Bounds
        globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);
        //
        // Viewing
        //
        view = new View();
        view.setPhysicalBody(new PhysicalBody());
        view.setPhysicalEnvironment(new PhysicalEnvironment());

        GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();
        try {
            jCanvas3D = new DistortStringReflectionJC3DOB(gCT, this);
        }
        catch (Exception e) {
            System.out.println("DistortStringReflection: JCanvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }

        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);
        //
        // BranchGraphs
        //
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();

        // ViewBranch

        TransformGroup viewTG = new TransformGroup();
        viewTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        ViewPlatform vp = new ViewPlatform();
        view.attachViewPlatform(vp);

        orbitBehInterim = new OrbitBehaviorInterim(jCanvas3D, viewTG, view, OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehInterim.setSchedulingBounds(globalBounds);
        orbitBehInterim.setClippingEnabled(true);

        Transform3D homeTransform = new Transform3D();
        homeTransform.setTranslation(new Vector3d(0.0, 0.45, 3.4));
        orbitBehInterim.setHomeTransform(homeTransform);
        orbitBehInterim.setHomeRotationCenter(new Point3d(0.0, 0.0, 0.0));
       
        DirectionalLight headLight = new DirectionalLight();
        headLight.setInfluencingBounds(globalBounds);

        viewTG.addChild(vp);
        viewTG.addChild(orbitBehInterim);
        viewTG.addChild(headLight);

        viewBranch.addChild(viewTG);

        // EnviBranch

        Background bg = new Background();
        bg.setApplicationBounds(globalBounds);
        bg.setColor(new Color3f(bgColor));

        enviBranch.addChild(bg);
    }

    // Finishing JCanvas3D (create PickCanvas if needed)
    private void createJCanvas3D() {
        
        // Arbitrary size
        Dimension dim = new Dimension(100, 100); 
        jCanvas3D.setPreferredSize(dim);
        jCanvas3D.setSize(dim);
        
        // Provide a parent !!
        jPanel = new JPanel(new BorderLayout());        
        jPanel.add(jCanvas3D, BorderLayout.CENTER);
        
        // Optional: less 'flickering' during resizing 
        jCanvas3D.setBackground(bgColor);

        // Off-screen Canvas3D of this lightweight JCanvas3D
        Canvas3D offCanvas3D = jCanvas3D.getOffscreenCanvas3D();
        if (offCanvas3D != null) {
            // View renders into the off-screen Canvas3D
            view.addCanvas3D(offCanvas3D);
        }
        else {
            System.out.println("DistortStringReflection: OffscreenCanvas3D = null !!");
            System.exit(0);
        }
    }
    
    // Derived from "org.jdesktop.j3d.examples.distort_glyph.DistortBehavior.java"
    // See license notice above
    private final class DistortBehavior extends Behavior implements GeometryUpdater {

        private static final int START          =   1;
        private static final int STOP           =   99;
        private WakeupCriterion startWakeup     =   null;
        private WakeupCriterion stopWakeup      =   null;       
        private WakeupCriterion frameWakeup     =   null;     
        private WakeupOr        stopFrameWakeup =   null;

        private int             numFrames       =   0;
        private int             frameNumber     =   0;
        private int             frame           =   0;
        
        private GeometryArray   geometryArray   =   null;

        private Vector3f        coord           =   new Vector3f();
        private float[]         origCoordArray  =   null;
        private float[]         copyCoordArray  =   null;
        private int             coordsLength    =   0;

        private Transform3D     t3              =   new Transform3D();

        DistortBehavior(GeometryArray geometry, int numFrames) {

            startWakeup = new WakeupOnBehaviorPost(this, START);
            stopWakeup  = new WakeupOnBehaviorPost(this, STOP);
            frameWakeup = new WakeupOnElapsedFrames(0);
            
            WakeupCriterion[] conditions = {stopWakeup, frameWakeup};
            stopFrameWakeup = new WakeupOr(conditions);

            this.numFrames = numFrames;

            geometryArray = geometry;

            origCoordArray = geometry.getCoordRefFloat();
            copyCoordArray = (float[])origCoordArray.clone();

            coordsLength = origCoordArray.length;
        }

        // Interface GeometryUpdater
        public void updateData(Geometry geometry) {

            t3.setIdentity();
            
            for (int n=0; n < coordsLength; n+=3) {

                coord.x = copyCoordArray[n];
                coord.y = copyCoordArray[n+1];
                coord.z = copyCoordArray[n+2];

                float spx = (float)Math.sin(frame *3f / 500);
                float spy = (float)Math.cos(frame *5f / 500);

                float px = coord.x - spx;
                float py = coord.y - spy;
                float pz = coord.z;

                float d = (float) Math.sqrt(px * px + py * py + pz * pz);

                t3.rotZ(d);
                t3.rotX(d*2);
                t3.rotY(d);
                t3.transform(coord);

                origCoordArray[n]   = coord.x;
                origCoordArray[n+1] = coord.y;
                origCoordArray[n+2] = coord.z;
            }
        }

        // Behavior
        
        @Override
        public void initialize() {
            wakeupOn(startWakeup);
        }

        @Override
        public void processStimulus(Enumeration criteria) {
            while (criteria.hasMoreElements()) {
                WakeupCriterion wakeUp = (WakeupCriterion) criteria.nextElement();

                if (wakeUp instanceof WakeupOnElapsedFrames) {
                    // modify the GeometryArray
                    frameNumber++;
                    frame++;

                    geometryArray.updateData(this);
                    
                    if (frameNumber > numFrames) {
                        // restart
                        frameNumber = 0;
                    }

                    wakeupOn(stopFrameWakeup);
                    return;
                }
                else if (wakeUp instanceof WakeupOnBehaviorPost) {
                    int postId = ((WakeupOnBehaviorPost)wakeUp).getTriggeringPostId();
                    if (postId == STOP) {
                        wakeupOn(startWakeup);
                    }
                    else if (postId == START) {
                        wakeupOn(stopFrameWakeup);
                    }
                    return;
                }
            }
        }
    }
}