
/*
 * LICENSE
 * 
 * JCanvas3DOBAbstract.java 
 * 
 * is a derived work of the Java 3D utility class "com.sun.j3d.exp.swing.JCanvas3D.java".
 * 
 * Redistribution and use are permitted according to the following license notice.
 * 
 * Version: 1.0
 * Date: 2009/03/09
 * 
 * Author:
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 * 
*/

/*
 * $RCSfile: com.sun.j3d.exp.swing.JCanvas3D.java,v $
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistribution of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistribution in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
 * USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
 * ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or
 * intended for use in the design, construction, operation or
 * maintenance of any nuclear facility.
 *
 * $Revision: 1.10 $
 * $Date: 2007/04/11 02:08:56 $
 * $State: Exp $
 * 
 *  @author: Frederic 'pepe' Barachant
 *
 */

package com.interactivemesh.j3d.testspace.jcanvas3d;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.GradientPaint;
import java.awt.Graphics;
import java.awt.Graphics2D;

import javax.media.j3d.GraphicsConfigTemplate3D;

import com.interactivemesh.j3d.community.gui.JCanvas3DOBAbstract;

/**
 * DistortStringReflectionJC3DOB.java
 *
 * Version: 2.0
 * Date: 2009/03/09
 * 
 * Author:
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * under consideration of the license notice above for the suberclass 'JCanvas3DOBAbstract',
 * but you may not modify, compile, or distribute this 'DistortStringReflectionJC3DOB.java'.
 *
 */
final class DistortStringReflectionJC3DOB extends JCanvas3DOBAbstract {

    //
    // Universe
    //
    
    private DistortStringReflection universe = null;
     
    //
    // Reflection
    //
    
    private AlphaComposite  alphaComposite  =   AlphaComposite.SrcOver; // default: SrcOver
    private GradientPaint   gradPaint       =   null;
    private float           reflectFraction =   1.0f; 
    private float           topOpacity      =   0.1f;
    private float           botOpacity      =   0.8f;
    private float[]         topCol          =   {0, 0, 0};
    private float[]         botCol          =   {0, 0, 0};
    // reflection height = reflectFraction * image height
    private int             reflectHeight   =   0;

    //
    // Frames per second
    //
    
    private int     updatesPerSecond    =   2;
    private int     elapsedFrames       =   0;
    private int     frameCounter        =   0;
    private long    startTimePaint      =   System.nanoTime()/1000000;
    private long    endTimePaint        =   0;
    private long    frameDuration       =   0;
            
    /**
     * Constructs and initializes a new DistortStringReflectionJC3DOB object that Java 3D
     * can render into, using the specified template.
     * The screen device is obtained from
     * <code>GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice()</code>,
     * which might not be the one you should use if you are
     * in a multiscreen environment.
     *
     * @param template The template that will be used to construct a
     *        GraphicsConfiguration. The stereo and doublebuffer properties
     *        are forced to UNNECESSARY.
     */
    DistortStringReflectionJC3DOB(GraphicsConfigTemplate3D template, 
                                   DistortStringReflection universe) {
        super(template);
        
        this.universe = universe;
    }
    
    //
    // Reflection API
    //
    
    public float getFraction() {
        return reflectFraction;
    }
    public void setFraction(float fraction) {
        reflectFraction = fraction;
        // Reflection setup: requires new off-screen buffers
        createOffScreenBuffer(getWidth(), getHeight());
    }
        
    public float getTopOpacity() {
        return topOpacity;
    }
    public void setTopOpacity(float opacity) {
        topOpacity = opacity;
        setGradient();
        
        this.repaint();
    }

    public float getBottomOpacity() {
        return botOpacity;
    }
    public void setBottomOpacity(float opacity) {
        botOpacity = opacity;
        setGradient();
        
        this.repaint();
    }
    
    public Color getTopColorRGB() {
        return new Color(topCol[0], topCol[1], topCol[2]);
    }
    public void setTopColorRGB(Color rgb) {
        rgb.getColorComponents(topCol);
        setGradient();
        
        this.repaint();
    }

    public Color getBottomColorRGB() {
        return new Color(botCol[0], botCol[1], botCol[2]);
    }
    public void setBottomColorRGB(Color rgb) {
        rgb.getColorComponents(botCol);
        setGradient();
        
        this.repaint();
    }

    private void setGradient() {
        gradPaint = new GradientPaint(
            0, 0,             new Color(topCol[0], topCol[1], topCol[2], topOpacity),              
            0, reflectHeight, new Color(botCol[0], botCol[1], botCol[2], botOpacity)); 
    }    
    
    //
    // Implement/Override JCanvas3DOBAbstract
    //
    
    /**
     * Creates new off-screen buffers of the given size. This method is called 
     * internally whenever this panel is added to a parent or is resized. 
     * <p>
     * Subclasses should call and/or override this method according to its
     * individual needs. In case of overriding calling
     * <code>super.createOffScreenBuffer(canvasWidth, canvasHeight)</code> 
     * has to be the last thing to do.</p>
     * @param width the width of the off-screen buffers to create
     * @param height the height of the off-screen buffers to create
     */
    @Override
    protected void createOffScreenBuffer(int panelWidth, int panelHeight) {
        
        // 1. Canvas size
        float floatHeight = (float)panelHeight / (1.0f + reflectFraction);
        int   intHeight   = (int)(floatHeight + 0.5f);
        
        reflectHeight = panelHeight - intHeight;
               
        // 2. GradientPaint
        setGradient();


        // At last !!
        super.createOffScreenBuffer(panelWidth, intHeight);
    }
    
    /**
     * Callback used to allow an overriding subclass to execute individual code
     * when new off-screen buffers were created. 
     * <p>
     * This method is called internally by the event-dispatching thread (EDT)
     * and should not be called by applications. 
     * </p>
     */
    @Override
    protected void offScreenBufferCreated() {      
        // Nothing to do
    }
    
    /**
     * Callback used to allow an overriding subclass to execute individual code
     * when the off-screen buffers were copied.
     * <p>
     * This method is called internally by the event-dispatching thread (EDT)
     * and should not be called by applications. 
     * </p>
     */
    @Override
    protected void offScreenBufferCopied() {
        // Nothing to do
    }
    
    /**
     * Flips and paints the result of the 3D rendering.
     *
     * @param g {@inheritDoc}
     */
    @Override
    public void paintComponent(Graphics g) {
     
        // paint background
        super.paintComponent(g); 
        
        // Ready for drawing ?
        if (!isReadyForDrawing()) {
            return;
        }
                        
        Graphics2D g2d = (Graphics2D)g;

        // Draw & flip scene image at the top
        g2d.drawImage(paintImage,
                  // destination in g2d: flip lowerY and upperY 
                  // dx1    dy1          dx2         dy2
                      0, imageHeight, imageWidth, 0,    
                  // source: the bottom part of the scene image
                  // sx1    sy1          sx2         sy2
                      0, 0,           imageWidth, imageHeight, null);
        
        // Reflection
        if (reflectHeight > 0) {
            // Draw mirrored scene image below, image is already yUp 
            g2d.drawImage(paintImage,
                      // destination in g2d
                      // dx1    dy1          dx2         dy2
                          0, imageHeight, imageWidth, getHeight(),    
                      // source: the upper part of the scene image
                      // sx1    sy1          sx2        sy2
                          0, 0,           imageWidth, reflectHeight, null);
                            
            // Move to mirrored scene image
            g2d.translate(0, imageHeight);
            
            // Set gradient
            g2d.setPaint(gradPaint);
            // Set composite
            g2d.setComposite(alphaComposite); 
            
            // Draw gradient according current AlphaComposite state
            // on top of the mirrored scene image
            g2d.fillRect(0, 0, imageWidth, reflectHeight);
        }
        
        // Frames per second

        frameCounter++;

        if (frameCounter >= elapsedFrames) {

            endTimePaint = System.nanoTime()/1000000;
            frameDuration = (endTimePaint-startTimePaint)/frameCounter;
            startTimePaint = endTimePaint;

            int fpsPaint = (int)(1000 / frameDuration);

            this.universe.updateFPS(fpsPaint);
            
            // Reset
            frameCounter = 0;
            elapsedFrames = (int)Math.max(1, ((fpsPaint + 0.5) / updatesPerSecond));
        }           
    }
}
