package com.interactivemesh.j3d.community.gui;

/*
 * LICENSE
 * 
 * FXCanvas3DMV API 1.1
 * 
 * This API is a derived work of the Java 3D utility class "com.sun.j3d.exp.swing.JCanvas3D.java"
 * 
 * and includes the files
 * 
 * - "com.interactivemesh.j3d.community.gui.FXCanvas3DMV.java"
 * - "com.interactivemesh.j3d.community.gui.FXCanvas3DMVControl.java" 
 * - "com.interactivemesh.j3d.community.gui.FXCanvas3DMVRepainter.java" 
 * 
 * 
 * Redistribution and use are permitted according to the following license notice.
 * 
 * Version: 1.1
 * Date: 2009/11/10
 * 
 * Author:
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 * 
*/
/*
 * $RCSfile: com.sun.j3d.exp.swing.JCanvas3D.java,v $
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistribution of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistribution in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
 * USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
 * ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or
 * intended for use in the design, construction, operation or
 * maintenance of any nuclear facility.
 *
 * $Revision: 1.10 $
 * $Date: 2007/04/11 02:08:56 $
 * $State: Exp $
 * 
 *  @author: Frederic 'pepe' Barachant
 *
 */

import java.awt.AWTEvent;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.GraphicsConfigTemplate;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Rectangle;

import java.awt.event.ComponentEvent;
import java.awt.event.FocusEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseWheelEvent;

import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;

import javax.media.j3d.Canvas3D;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.ImageComponent2D;
import javax.media.j3d.RestrictedAccessException;
import javax.media.j3d.Screen3D;

import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import javax.swing.event.AncestorEvent;
import javax.swing.event.AncestorListener;

import com.sun.j3d.exp.swing.impl.AutoOffScreenCanvas3D;

/**
 * <p>
 * This class provides Java 3D rendering into a lightweight component 
 * for several simultaneous views controlled by a FXCanvas3DMVControl object.
 * </p> 
 *
 * @since Java 3D 1.5
 * 
 * @see FXCanvas3DMVControl
 * @see FXCanvas3DMVRepainter
 */
public class FXCanvas3DMV extends JPanel implements AncestorListener {

    /** size of a pixel */
    private static double METERS_PER_PIXEL = 0.0254 / 90.0;
    
    /** the graphics configuration used for this canvas */
    private GraphicsConfiguration   graphicsConfig  =   null;

    /** The canvas that is linked to the component. */
    private OffscreenCanvas3D       canvas          =   null;
        
    private long                    wakeupEventMasks =  0;

    //
    // Double Offscreen Buffer
    //
    private BufferedImage       paintImage          =   null;
    private int[]               paintIntBuffer      =   null;
    private int[]               renderIntBuffer     =   null;
    private int                 imageWidth          =   0;      
    private int                 imageHeight         =   0;      
    
    // FXCanvas3DMVControl
    private FXCanvas3DMVControl mvControl           =   null;     
    private int                 canvasIndex         =   -1;
    

/*
private long startTimePaint = 0;
private long endTimePaint = 0;

private long startTimePre = 0;
private long startTimeSwap = 0;
private long endTimeSwap = 0;
private long startTimeRender = 0;
private long endTimeRender = 0;
*/
    /**
     * Constructs and initializes a new FXCanvas3DMV object that Java 3D
     * can render into. The screen device is obtained from
     * <code>GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice()</code>,
     * which might not be the one you should use if you are in a multiscreen environment.
     * The FXCanvas3DMV is constructed using the following default parameters:<br>
     * double buffer enable : false<br>
     * stereo enable : false<br>
     */
    public FXCanvas3DMV() {
        this(null, null);
    }

    /**
     * Constructs and initializes a new FXCanvas3DMV object that Java 3D
     * can render into, using the specified graphics device.
     *
     * @param device the screen graphics device that will be used to construct
     *        a GraphicsConfiguration.
     */
    public FXCanvas3DMV(GraphicsDevice device) {
        this(null, device);
    }

    /**
     * Constructs and initializes a new FXCanvas3DMV object that Java 3D
     * can render into, using the specified template.
     * The screen device is obtained from
     * <code>GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice()</code>,
     * which might not be the one you should use if you are
     * in a multiscreen environment.
     *
     * @param template The template that will be used to construct a
     *        GraphicsConfiguration. The stereo and doublebuffer properties
     *        are forced to UNNECESSARY.
     */
    public FXCanvas3DMV(GraphicsConfigTemplate3D template) {
        this(template, null);
    }
    
    /**
     * Constructs and initializes a new FXCanvas3DMV object that Java 3D
     * can render into, using the specified template and graphics device.
     *
     * @param template The template that will be used to construct a
     *        GraphicsConfiguration. The stereo and doublebuffer properties
     *        are forced to UNNECESSARY.
     * @param device the screen graphics device that will be used to construct
     *        a GraphicsConfiguration in conjunction with the template.
     */
    FXCanvas3DMV(GraphicsConfigTemplate3D template, GraphicsDevice device) {
        
        if (template == null) {
            template = new GraphicsConfigTemplate3D();
        }
        // Force double-buffer and stereo to UNNECESSARY
        template.setStereo(GraphicsConfigTemplate.UNNECESSARY);
        template.setDoubleBuffer(GraphicsConfigTemplate.UNNECESSARY);

        if (device == null) {
            graphicsConfig = GraphicsEnvironment.getLocalGraphicsEnvironment().
                                getDefaultScreenDevice().getBestConfiguration(template);
        }
        else {       
            graphicsConfig = device.getBestConfiguration(template);
        }
        
        this.addAncestorListener(this);
        this.setDoubleBuffered(false);

        // so that key events can be received.
        this.setFocusable(true);
        // A ComponentListener is added to the offscreen Canvas3D by default !
        this.enableEvents(AWTEvent.COMPONENT_EVENT_MASK);

    }
    
    //
    // Multiple Views
    //
    
    void setFXCanvas3DMVControl(FXCanvas3DMVControl control, int index) {
        mvControl = control;
        canvasIndex = index;
    }
    
    int getFXCanvas3DMVIndex() {
        return canvasIndex;
    }
    
    boolean isCanvas3DCreated() {
        return (canvas != null);
    }
    
    void startStopRenderer(boolean start) {
        if (canvas == null)
            return;
        
        if (start)
            canvas.startRenderer();
        else
            canvas.stopRenderer();
    }
    
    //
    // JavaFX - Java 3D bridge
    //
    
    /** 
     * Copies the 3D rendering image into the 2D painting image
     * while the J3D-Renderer thread is waiting.
     * To be called from the FXCanvas3DMVControl object.
     * 
     */
    void copyOffScreenBuffer() {      
        if (paintImage != null) {        
            // Copy 3D rendering image into 2D painting image
            System.arraycopy(renderIntBuffer, 0, paintIntBuffer, 0, renderIntBuffer.length);
        }
    }
    
    //
    // Interface AncestorListener
    //
    
    /**
     * {@inheritDoc}
     *
     * @param event {@inheritDoc}
     * @throws IllegalStateException if no FXCanvas3DMVControl object is set
     */
     public void ancestorAdded(AncestorEvent event) {
        Dimension sz = getSize();

        if (sz.width < 1) {
            sz.width = 100;
        }

        if (sz.height < 1) {
            sz.height = 100;
        }

        if (mvControl == null)
            throw new IllegalStateException("No FXCanvas3DMVControl object is set !!");

        mvControl.createOffScreenCanvas(canvasIndex, sz.width, sz.height);
        
        canvas.addNotifyFlag = true; // make it so that i can call addNotify() without being rejected.
        canvas.addNotify();
    }

    /**
     * {@inheritDoc}
     *
     * @param event {@inheritDoc}
     */
    public void ancestorMoved(AncestorEvent event) {
    }

    /**
     * {@inheritDoc}
     *
     * @param event {@inheritDoc}
     */
    public void ancestorRemoved(AncestorEvent event) {
        canvas.removeNotify();
    }

    //
    // Java 3D specific methods
    //
    
    /**
     * Returns the off-screen heavyweight canvas of this lightweight component.
     *
     * @return the heavyweight canvas that lies in the deepness of this Component.
     * 
     * @throws IllegalStateException if no FXCanvas3DMVControl object is set
     */
    public Canvas3D getOffscreenCanvas3D() {
        if (this.canvas == null) {
            if (mvControl == null)
                throw new IllegalStateException("No FXCanvas3DMVControl object is set !!");
            mvControl.createOffScreenCanvas(canvasIndex, getWidth(), getHeight());
        }

        return this.canvas;
    }
    
    /**
     * {@inheritDoc}
     *
     * @param x {@inheritDoc}
     * @param y {@inheritDoc}
     * @param width {@inheritDoc}
     * @param height {@inheritDoc}
     * @throws IllegalStateException if no FXCanvas3DMVControl object is set
     */
    @Override
    public void setBounds(int x, int y, int width, int height) {

        super.setBounds(x, y, width, height);

        if (mvControl == null)
            throw new IllegalStateException("No FXCanvas3DMVControl object is set !!");
                
        if (canvas != null) {
            if (canvas.getWidth() == width && canvas.getHeight() == height) {
                return;
            }
        }
        
        mvControl.createOffScreenCanvas(canvasIndex, width, height);    
    }

    // Called from FXCanvas3DMVControl only !!!!!
    /**
     * Creates a heavyweight canvas and initializes it, or changes the
     * size of the current one if present. Current heavyweight canvas is
     * changed only if size is different from the actual one. 
     * 
     * No canvas is created if this component has no parent, that is, 
     * was not added to a container.
     *
     * @param width the width of the canvas to create.
     * @param height the height of the canvas to create.
     */
    final void createOffScreenCanvas(int width, int height) {
       
        if (getParent() == null) {
            return;
        }
        
        final int w = Math.max(1, width);
        final int h = Math.max(1, height);
        
        if (canvas == null) {
            canvas = new OffscreenCanvas3D(this.graphicsConfig);
        } 
        else {
            if (canvas.getWidth() == w && canvas.getHeight() == h)
                return;
        }
        
        // Create 'Double OffScreen Buffer'
        
        if (SwingUtilities.isEventDispatchThread()) {
            canvas.createOffScreenBuffer(graphicsConfig.getBounds(), w, h);
        }
        else {
            SwingUtilities.invokeLater(new Runnable() { 
                public void run() {
                    canvas.createOffScreenBuffer(graphicsConfig.getBounds(), w, h);
                }
            });
        }
    }
    
    /**
     * Flips and paints the result of the 3D rendering.
     *
     * @param g {@inheritDoc}
     */
    @Override
    public void paintComponent(Graphics g) {
        
        super.paintComponent(g); // paint background
        
        if (paintImage == null || !canvas.isRendererRunning()) {
            return;
        }

        // Flip and draw paintImage 
        g.drawImage(paintImage, 
                // destination: Graphics g, flip lowerY and upperY 
                // dx1 dy1          dx2         dy2
                    0, imageHeight, imageWidth, 0,
                 // source: BufferedImage paintImage
                 // sx1 sy1         sx2         sy2
                    0, 0,           imageWidth, imageHeight, null);             
    }

    // WakeupOnAWTEvent: COMPONENT_EVENT_MASK, FOCUS_EVENT_MASK, 
    //                   KEY_EVENT_MASK
    //                   MOUSE_EVENT_MASK, MOUSE_MOTION_EVENT_MASK, MOUSE_WHEEL_EVENT_MASK
    /**
     * Sets the AWT event masks for which this FXCanvas3DMV object 
     * will call the corresponding process-methods on the underlying heavyweight canvas.
     * <p>
     * This is required only when Java 3D WakeupOnAWTEvents are specified in subclasses of Behavior. 
     * <p>
     * No event mask is set per default.
     * 
     * @param eventMasks ored AWT event masks: <code>AWTEvent.COMPONENT_EVENT_MASK,
     * AWTEvent.FOCUS_EVENT_MASK, AWTEvent.KEY_EVENT_MASK, AWTEvent.MOUSE_EVENT_MASK,
     * AWTEvent.MOUSE_MOTION_EVENT_MASK, AWTEvent.MOUSE_WHEEL_EVENT_MASK</code>.
     * 
     * @since 1.1
     */
    public void enableWakeupOnAWTEvents(long eventMasks) {
        
        if ((eventMasks & AWTEvent.COMPONENT_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.COMPONENT_EVENT_MASK;
        if ((eventMasks & AWTEvent.FOCUS_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.FOCUS_EVENT_MASK;
        if ((eventMasks & AWTEvent.KEY_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.KEY_EVENT_MASK;
        if ((eventMasks & AWTEvent.MOUSE_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.MOUSE_EVENT_MASK;
        if ((eventMasks & AWTEvent.MOUSE_MOTION_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.MOUSE_MOTION_EVENT_MASK;
        if ((eventMasks & AWTEvent.MOUSE_WHEEL_EVENT_MASK) != 0)
            wakeupEventMasks |= AWTEvent.MOUSE_WHEEL_EVENT_MASK;        
        /*
        Enables the events defined by the specified event mask parameter 
        to be delivered to this component.

        Event types are automatically enabled when a listener for that event type 
        is added to the component.

        This method only needs to be invoked by subclasses of Component 
        which desire to have the specified event types delivered 
        to processEvent regardless of whether or not a listener is registered. 
        */
        this.enableEvents(wakeupEventMasks);
    }
    /**
     * Disables calling event process-methods on the underlying heavyweight canvas
     * for the specified AWT event masks.
     * @param eventMasks ored AWT event masks: <code>AWTEvent.COMPONENT_EVENT_MASK,
     * AWTEvent.FOCUS_EVENT_MASK, AWTEvent.KEY_EVENT_MASK, AWTEvent.MOUSE_EVENT_MASK,
     * AWTEvent.MOUSE_MOTION_EVENT_MASK, AWTEvent.MOUSE_WHEEL_EVENT_MASK</code>
     * 
     * @since 1.1
     */
    public void disableWakeupOnAWTEvents(long eventMasks) {
        wakeupEventMasks &= ~eventMasks;
        this.disableEvents(eventMasks);
    }
    
    //
    // Override Component's processXXX
    //
    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processComponentEvent(ComponentEvent e) {
        super.processComponentEvent(e);
//System.out.println("FXCanvas3DSB processComponentEvent = " + e.getSource().getClass());        
        if ((wakeupEventMasks & AWTEvent.COMPONENT_EVENT_MASK) != 0) {
            Object src = e.getSource();
            e.setSource(canvas);
            canvas.processComponentEvent(e);
            e.setSource(src);
        }
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processFocusEvent(FocusEvent e) {
        super.processFocusEvent(e);

        if ((wakeupEventMasks & AWTEvent.FOCUS_EVENT_MASK) != 0) {
            Object src = e.getSource();
            e.setSource(canvas);
            canvas.processFocusEvent(e);
            e.setSource(src);
        }
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processKeyEvent(KeyEvent e) {
        super.processKeyEvent(e);

        if ((wakeupEventMasks & AWTEvent.KEY_EVENT_MASK) != 0) {
            Object src = e.getSource();
            e.setSource(canvas);
            canvas.processKeyEvent(e);
            e.setSource(src);
        }
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processMouseEvent(MouseEvent e) {
        super.processMouseEvent(e);

        if ((wakeupEventMasks & AWTEvent.MOUSE_EVENT_MASK) != 0) {
            Object src = e.getSource();
            e.setSource(canvas);
            canvas.processMouseEvent(e);
            e.setSource(src);
        }
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processMouseMotionEvent(MouseEvent e) {
        super.processMouseMotionEvent(e);

        if ((wakeupEventMasks & AWTEvent.MOUSE_MOTION_EVENT_MASK) != 0) {
            Object src = e.getSource();
            e.setSource(canvas);
            canvas.processMouseMotionEvent(e);
            e.setSource(src);
        }
    }

    /**
     * Redirects event to canvas and to superclass.
     *
     * @param e {@inheritDoc}
     */
    @Override
    protected void processMouseWheelEvent(MouseWheelEvent e) {
        super.processMouseWheelEvent(e);

        if ((wakeupEventMasks & AWTEvent.MOUSE_WHEEL_EVENT_MASK) != 0) {
            Object src = e.getSource();
            e.setSource(canvas);
            canvas.processMouseWheelEvent(e);
            e.setSource(src);
        }
    }

    /**
     * This class is the internal Canvas3D that is used and sent to
     * Java 3D. It is remote controlled through FXCanvas3DMV and is modified to be
     * able to tell the lightweight component when refreshes are needed.
     */
    private final class OffscreenCanvas3D extends    Canvas3D 
                                          implements AutoOffScreenCanvas3D {

        // Render target
        private BufferedImage renderImage = null;

        /**
         * Flag used to sort a call to addnotify() from user and
         * from the lightweight component. Lightweight component calls
         * addNotify() so that the rendering begins and uses normal routines,
         * but this is a method that user must not call.
         */
        private boolean addNotifyFlag = false;
        
        /**
         * Creates a new instance of FXCanvas3DMV.
         * 
         * @param graphicsConfiguration The graphics configuration to be used.
         * @param lwCanvas the lightweight canvas that is linked to that
         *        heavyweight canvas.
         */
        private OffscreenCanvas3D(GraphicsConfiguration graphicsConfiguration) {
            super(graphicsConfiguration, true);
        }
        
        // FXCanvas3DMVControl takes care about thread synchronisation and start/stopRenderer
        // To be called on the EDT only !!!!!!!!!!!!!!
        /**
         * Creates an offscreen buffer to be attached to the heavyweight
         * buffer. Buffer is created 'byreference'
         *
         * @param width the width of the buffer.
         * @param height the height of the buffer.
         */
        private void createOffScreenBuffer(Rectangle screenRect, int width, int height) {

            // Compute physical dimensions of the screen
            // Fix to Issue : 433 - JCanvas3D crashed when using jogl pipe.
            int screenWidth = (int)screenRect.getWidth();
            int screenHeight = (int)screenRect.getHeight();
            Screen3D screen3D = this.getScreen3D();
            screen3D.setSize(screenWidth, screenHeight);
            screen3D.setPhysicalScreenWidth(((double) screenWidth) * METERS_PER_PIXEL);
            screen3D.setPhysicalScreenHeight(((double) screenHeight) * METERS_PER_PIXEL);
            
            if (renderImage != null)
                renderImage.flush();
            if (paintImage != null)
                paintImage.flush();
            
            // OffScreenBuffer: byReference & yUp !! V 2.0
            renderImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
            ImageComponent2D image = new ImageComponent2D(ImageComponent2D.FORMAT_RGBA, renderImage, true, true);
            image.setCapability(ImageComponent2D.ALLOW_IMAGE_READ);
            image.setCapability(ImageComponent2D.ALLOW_IMAGE_WRITE);
            
            paintImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);

            paintIntBuffer  = ((DataBufferInt)paintImage.getRaster().getDataBuffer()).getData();
            renderIntBuffer = ((DataBufferInt)renderImage.getRaster().getDataBuffer()).getData();
            
            imageWidth  = width;
            imageHeight = height;
            
            // Offscreen rendering might occur even if the renderer is stopped. 
            // For that reason, we are waiting for a hypothetical offscreen render 
            // to finish before setting offscreen rendering.
            // Otherwise, rendering will stop with an exception.
            this.waitForOffScreenRendering();               // placed here in V 2.0

            try {
                // EDT will wait (sleep) until exchange of offscreen buffer is done 
                this.setOffScreenBuffer(image);                
            }
            catch (RestrictedAccessException e) {
                // Try again! Risk of endless loop ?  TODO 
                createOffScreenBuffer(screenRect, width, height); // V 2.0
                System.out.println("Repeat : createOffScreenBuffer(screenRect, width, height) !!");
            }
        }
        
        /**
         * {@inheritDoc}
         */
        @Override
        public void addNotify() {
            if (addNotifyFlag) {
                addNotifyFlag = false;
                super.addNotify();                
            } 
            else {
                throw new UnsupportedOperationException("User must not call this method !!");
            }
        }

        /**
         * Normally, returns the parent of that component. As the
         * canvas ought to never be added to any component, it has no parent.
         * Java 3D expects it to have a parent for some operations, so we in
         * fact cheat it by returning the parent of the lightweight component.
         *
         * @return the parent of the lightweight component, if any. Returns
         *         null if it has no parent.
         */
        @Override 
        public Container getParent() {
            return FXCanvas3DMV.this.getParent();
        }

        /**
         * Done by FXCanvas3DMVControl.
         */
        @Override
        public void postSwap() {            
            mvControl.postSwapFXCanvas3DMV(canvasIndex);
        }
        
        // ----------------- Java 3D engine ------------------------------------
        
        // WakeupOnAWTEvent: COMPONENT_EVENT_MASK, FOCUS_EVENT_MASK, 
        //                   KEY_EVENT_MASK
        //                   MOUSE_EVENT_MASK, MOUSE_MOTION_EVENT_MASK, MOUSE_WHEEL_EVENT_MASK

        // Canvas3D / WakeupOnAWTEvent
        // EventCatcher: FocusListener, KeyListener, 
        //               MouseListener, MouseMotionListener, MouseWheelListener
        
        // Canvas3D, all ancestors, Window
        // EventCatcher: ComponentListener
        
        // Canvas3D, all ancestors, Window
        // CanvasViewEventCatcher: ComponentAdapter (componentResized/componentMoved) 

        // Window
        // EventCatcher: WindowListener
        
        // Hint: Offscreen-Canvas3D: componentMoved, componentHidden, componentShown
        //                           are not relevant
        //                           componentResized is called due to 
        //                                            setSize-call in setOffScreenBuffer
        
        // ---------------------------------------------------------------------

        //
        // Override Component's processXXX
        //
        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processComponentEvent(ComponentEvent e) {
            super.processComponentEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processFocusEvent(FocusEvent e) {
            super.processFocusEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processKeyEvent(KeyEvent e) {
            super.processKeyEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processMouseEvent(MouseEvent e) {
            super.processMouseEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processMouseMotionEvent(MouseEvent e) {
            super.processMouseMotionEvent(e);
        }

        /**
         * Overriden so that the JComponent can access it.
         *
         * @param e {@inheritDoc}
         */
        @Override
        protected void processMouseWheelEvent(MouseWheelEvent e) {
            super.processMouseWheelEvent(e);
        }
    }
}
