package com.interactivemesh.j3d.community.gui;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

import javax.media.j3d.Canvas3D;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;

import javax.swing.SwingUtilities;

/*
 * LICENSE
 * 
 * FXCanvas3DMV API 1.1
 * 
 * This API is a derived work of the Java 3D utility class "com.sun.j3d.exp.swing.JCanvas3D.java"
 * 
 * and includes the files
 * 
 * - "com.interactivemesh.j3d.community.gui.FXCanvas3DMV.java"
 * - "com.interactivemesh.j3d.community.gui.FXCanvas3DMVControl.java" 
 * - "com.interactivemesh.j3d.community.gui.FXCanvas3DMVRepainter.java" 
 * 
 * 
 * Redistribution and use are permitted according to the following license notice.
 * 
 * Version: 1.1
 * Date: 2009/11/10
 * 
 * Author:
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 * 
*/
/*
 * $RCSfile: com.sun.j3d.exp.swing.JCanvas3D.java,v $
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistribution of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistribution in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
 * USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
 * ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or
 * intended for use in the design, construction, operation or
 * maintenance of any nuclear facility.
 *
 * $Revision: 1.10 $
 * $Date: 2007/04/11 02:08:56 $
 * $State: Exp $
 * 
 *  @author: Frederic 'pepe' Barachant
 *
 */
/**
 * A FXCanvas3DMVControl object controls repainting and resizing of the 
 * FXCanvas3DMV objects. As soon as Java 3D has finished rendering into 
 * all attached and active FXCanvas3DMV objects for a single frame 
 * the FXCanvas3DMVControl object requests JavaFX to copy the image data 
 * and to repaint all corresponding SwingComponent objects in a single loop. 
 * 
 * @see FXCanvas3DMV
 * @see FXCanvas3DMVRepainter
 */
public class FXCanvas3DMVControl {
    
    //
    // 
    //
    private FXCanvas3DMV[]      fxCanvas3DMVs       =   null;
    private int                 fxCanvas3DLength    =   0;
    private int                 fxCancvas3DMVCount  =   0;    
    private int                 fxCanvas3DMVDone    =   0;
    
    private boolean[]           canvasToRender      =   null; 
    
    //
    // Lock - Conditions
    //
    private ReentrantLock       imageAccessLock     =   null;
    private Condition           imageCopyCondition  =   null;
    private Condition           imageResizeCondition =  null;
    private volatile boolean    isImageCopied       =   false;
    private volatile boolean    isRendererLocked    =   false;
    private volatile boolean    isResizing          =   false;

    private long                swapTimeCurr        =   0;
    private long                swapTimeLast        =   0;

    //
    // FXCanvas3D Repainter
    //
    private FXCanvas3DMVRepainter repainter         =   null;
    
    // Call repaint on EDT
    private Runnable            repaintCall         =   null;
    
    // Don't wait longer than 200 ms for repainting
    private static final int LIMIT_WAIT = 200000000;
    private static final int LIMIT_SWAP = 400000000; // 400 ms
    
/*    
private long startTimeRender = 0;
private long endTimeRender = 0;
*/


    /**
     * Constructs and initializes a new FXCanvas3DMVControl object for the given
     * FXCanvas3DMV objects.
     *
     * @param fxCanvas3Ds FXCanvas3DMV objects to manage 
     * 
     */
    public FXCanvas3DMVControl(FXCanvas3DMV[] fxCanvas3Ds) {
        
        // Sync image access
        imageAccessLock      = new ReentrantLock();
        imageCopyCondition   = imageAccessLock.newCondition();
        imageResizeCondition = imageAccessLock.newCondition();
        
        repaintCall = new RepaintCall();
        
        fxCanvas3DLength = fxCanvas3Ds.length;
        fxCanvas3DMVs = new FXCanvas3DMV[fxCanvas3DLength];
        canvasToRender = new boolean[fxCanvas3DLength];
        
        for (int i=0; i < fxCanvas3DLength; i++) {
            fxCanvas3DMVs[i] = fxCanvas3Ds[i];
            fxCanvas3DMVs[i].setFXCanvas3DMVControl(this, i);
        }
    }
    
    //
    // JavaFX - Java 3D bridge
    //
    
    /** 
     * Sets the FXCanvas3DMVRepainter object for this FXCanvas3DMVControl.
     * 
     * @param painter JavaFX class instance
     */
    public void setRepainter(FXCanvas3DMVRepainter painter) {
        repainter = painter;
    }
    
    // Called from a single FXCanvas3DMV on the EDT for synchronized re/sizing 
    // of its offscreen buffer
    void createOffScreenCanvas(int index, int width, int height) {
       try {
           imageAccessLock.lock();
           
           // Setting offscreen buffer requires that the J3D-Renderer thread isn't blocked
           // As we are on the EDT, FXCanvas3DComp/FXCanvas3DRepainter can't release the J3D-Renderer
           // So, it's done here. We are waiting until this has happened.
           while (isRendererLocked) {
               
                isImageCopied = true;
                imageCopyCondition.signal();
                
                try {
                   imageResizeCondition.await();
                }
                catch (InterruptedException e) {
                }
            }
               
           isResizing = true;
           
           fxCanvas3DMVs[index].startStopRenderer(false); 
           fxCanvas3DMVs[index].createOffScreenCanvas(width, height);
           fxCanvas3DMVs[index].startStopRenderer(true);
       }
       finally  {
           isResizing = false;
           imageAccessLock.unlock();
       }
    }
    
    /**
     * Convenient method for rezising all FXCanvas3DMV objects to the same size.
     * 
     * @param width the width of the canvas
     * @param height the height of the canvas
     */
    public void setFXCanvas3DMVSize(int width, int height) {
        
        try {
            imageAccessLock.lock();
            
            // Setting offscreen buffer requires that the J3D-Renderer thread isn't blocked
            // As we are on the EDT, FXCanvas3DComp/FXCanvas3DRepainter can't release the J3D-Renderer
            // So, it's done here. We are waiting until this has happened.
            while (isRendererLocked) {
                
                isImageCopied = true;
                imageCopyCondition.signal();
                 
                try {
                    imageResizeCondition.await();
                }
                catch (InterruptedException e) {
                }
            }
            
            isResizing = true;
                
            for (FXCanvas3DMV c3dmv : fxCanvas3DMVs) {
                c3dmv.startStopRenderer(false);
            }

            for (FXCanvas3DMV c3dmv : fxCanvas3DMVs) {
                c3dmv.createOffScreenCanvas(width, height);
            }
            
            for (FXCanvas3DMV c3dmv : fxCanvas3DMVs) {
                c3dmv.startStopRenderer(true);
            }
        }
        finally  {
            isResizing = false;
            imageAccessLock.unlock();
        }        
    }
    
    /** 
     * Copies the 3D rendering images into the 2D painting images
     * while the J3D-Renderer thread is waiting. 
     * Initiates repainting of all 2D painting images.
     * To be called from the JavaFX painting loop.
     * 
     */
    public void copyBuffersAndRepaint() {      
        
        try {      
            imageAccessLock.lock();
            
            for (int i=0; i < fxCanvas3DLength; i++) {
                if (canvasToRender[i]) {
                    fxCanvas3DMVs[i].copyOffScreenBuffer();
                }
            }
                
            // Notify the waiting J3D-Renderer thread.
            isImageCopied = true;
            imageCopyCondition.signal();        
        }
        finally {
            imageAccessLock.unlock();
        }
        
        for (int i=0; i < fxCanvas3DLength; i++) {
            if (canvasToRender[i]) {
                fxCanvas3DMVs[i].repaint();
            }
        }
    }
    
    // Synchronize repainting of all active FXCanvas3DMVs
    // Called from each active FXCanvas3DMV on the J3D-Renderer thread
    void postSwapFXCanvas3DMV(int canvasIndex) {
        
        if (isResizing)
            return;
        
        // Eliminate single swap call, restart 'counting'
        swapTimeCurr = System.nanoTime();
        if ( (swapTimeCurr - swapTimeLast) > LIMIT_SWAP) {
            fxCanvas3DMVDone = 0;
//System.out.println("FXCanvas3DMVControl postSwapFXCanvas3DMV : (swapTimeLast - swapTimeCurr) > LIMIT_SWAP !!!");
        }        
        swapTimeLast = swapTimeCurr;

        
        // First postSwap call: check which/how many canvas will be rendered
        if (fxCanvas3DMVDone == 0) {
            
            fxCancvas3DMVCount = 0;
            
            Canvas3D canvas3D = null;
            FXCanvas3DMV canvas3DMV = null;
            
            for (int i=0; i < fxCanvas3DLength; i++) {
                if (i == canvasIndex) {
                    canvasToRender[i] = true;
                    fxCancvas3DMVCount++;
                }
                else {
                    
                    canvas3DMV = fxCanvas3DMVs[i];
                    
                    int index = canvas3DMV.getFXCanvas3DMVIndex();                
                    canvasToRender[index] = false; // default
                    
                    if (canvas3DMV.isCanvas3DCreated()) {
                        
                        canvas3D = canvas3DMV.getOffscreenCanvas3D();
                        View view = canvas3D.getView();
                                            
                        // isViewRunning() can't be called from Canvas3D render callback  !!
                        
                        if (canvas3D.isRendererRunning() && view != null) {
                            ViewPlatform vp = view.getViewPlatform();
                            if (vp != null && vp.isLive()) {
                                
                                if (canvas3DMV.isShowing()) { // TODO                           
                                    canvasToRender[index] = true;
                                    fxCancvas3DMVCount++;
                                }
                            }
                        }
                    }
                }
            }
        }
        
        fxCanvas3DMVDone++;
        
        // All swaped ?
        if (fxCanvas3DMVDone < fxCancvas3DMVCount)
            return;
        
        // Now initialize common repainting of all active FXCanvas3DMVs
        
        fxCanvas3DMVDone = 0;
        
        try {
            imageAccessLock.lock();
                                
            isImageCopied = false;                
            isRendererLocked = true;
                           
            // calling 'repainter.repaintFXCanvas3D()' on
            // EDT per 'SwingUtilities.invokeLater' 
            // results in smoother mouse-navigation
            
            // Repainting in JavaFX painting loop on EDT                    
            SwingUtilities.invokeLater(repaintCall);
            
            // Wait until image is copied
            while (!isImageCopied) { 
                
                try {                           
                    imageCopyCondition.awaitNanos(LIMIT_WAIT);   // Don't wait for ever
                    isImageCopied = true;                        // and release yourself
                }
                catch (InterruptedException e) {
                }
            }

            isRendererLocked = false;
            imageResizeCondition.signal();
        }
        finally  {                      
           imageAccessLock.unlock();
        }                                     
    }
    
    private final class RepaintCall implements Runnable {
        public void run() {
            if (repainter != null)
                repainter.repaintFXCanvas3DMV();
        }
    }
}
