package com.interactivemesh.j3d.testspace.jfx.charcubesb;

import java.lang.Character;  

import com.javafx.preview.control.Menu;
import com.javafx.preview.control.MenuItem;
import com.javafx.preview.control.PopupMenu;
import com.javafx.preview.control.RadioMenuItem;

import javafx.scene.Group;
import javafx.scene.Scene;

import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.Separator;
import javafx.scene.control.Slider;
import javafx.scene.control.Toggle;
import javafx.scene.control.ToggleGroup;
import javafx.scene.control.Tooltip;

import javafx.scene.effect.Bloom;
import javafx.scene.effect.GaussianBlur;
import javafx.scene.effect.SepiaTone;

import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

import javafx.scene.layout.LayoutInfo;
import javafx.scene.layout.Priority;
import javafx.scene.layout.Tile;
import javafx.scene.layout.VBox;

import javafx.scene.paint.Color;
import javafx.scene.paint.LinearGradient;
import javafx.scene.paint.Stop;

import javafx.scene.shape.Rectangle;

import javafx.scene.text.Font;
import javafx.scene.text.FontWeight;

import javafx.stage.Screen;
import javafx.stage.Stage;

import javafx.util.Math;

/**
 * CharacterCubeMain.fx
 * 
 * Version: 5.1
 * Date: 2010/09/19
 *
 * Copyright (c) 2009-2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'CharacterCubeMain.fx'.
 *
 */

// TODO : How to check for application versus applet ??
// API : getArgument("javafx.applet") will return null if not running as an applet
// API : This is an experimental facility, that may be changed in future versions.
var isApplication: Boolean = (FX.getArgument("javafx.applet") == null);

// Frame
var stage: Stage;

// FXCanvas3D
def fxCanvas3DComp: FXCanvas3DComp = FXCanvas3DComp {
    // Resizing
    layoutInfo: LayoutInfo {
        width: bind Math.max(10, stage.scene.width - controlVBox.layoutBounds.width - 3 * border);
        height: bind Math.max(10, stage.scene.height - 2 * border);
    }
}

// UniverseFX
def universeFX = CharacterCubeUniverseFX {
    // Callback of AsyncOperation
    initUniverse: function(universe: CharacterCubeUniverse): Void {
        // Finish FXCanvas3DComp
        fxCanvas3DComp.initFXCanvas3D(universe);
        // Show frame
        stage.visible = true;
    }
    // Update call from universe
    updateUnicodeCodePoint: function(codepoint: Integer): Void {
        codePointCharLabel.text = String.valueOf(Character.toChars(codepoint));
        codePointIntLabel.text = String.valueOf(codepoint);
    }
    // Change rotation speed
    rotationValue: bind rotationSlider.value as Integer
    // Change cube transparency in universe
    transValue: bind cubeTransSlider.value as Integer
}

// Design

def backgroundColor = Color.color(0.0, 0.4, 0.8);

def screenHeight: Number = Screen.primary.bounds.height;
def screenDPI: Number = Screen.primary.dpi;

// screenHeight >= 1200
var textFontSize = 18;
var titleFontSize = 34;
var sceneHeight = 900;
var controlSpace = 10;
var border: Number = 50;

// screenHeight  < 1024
if (screenHeight < 1024) {
    textFontSize = 14;
    titleFontSize = 28;
    sceneHeight = 700;
    controlSpace = 6;
    border = 30;
}
// 1024 <= screenHeight < 1200
else if (screenHeight < 1200) {
    textFontSize = 16;
    titleFontSize = 30;
    sceneHeight = 800;
    controlSpace = 8;
    border = 40;
}

// Bug RT-9312, since 1.3.1
// Logical fonts (Dialog, Serif, etc.) overruns, although text doesn't exceed the available space. 

def titleFont = Font.font("Amble Cn", FontWeight.REGULAR, titleFontSize);
def textFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize);
def charFont = Font.font("Lucida Sans", FontWeight.REGULAR, textFontSize + 2);
def charLargeFont = Font.font("Lucida Sans", FontWeight.REGULAR, titleFontSize*6);
def menuFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize);
def tipFont = Font { size: textFontSize};

//
// Controls
//
class PopupSubMenu extends Menu {
    override var font = menuFont
}
class PopupMenuItem extends MenuItem {
    override var font = menuFont
}
class PopupRadioMenuItem extends RadioMenuItem {
    override var font = menuFont
}

class CubeTooltip extends Tooltip {
    override var font = tipFont;
}

def fillLayoutInfo =  LayoutInfo{hfill: true vfill: true hgrow: Priority.ALWAYS};

class BaseLabel extends Label {
    override var styleClass = "cube-label";
    override var font = textFont;
    override var layoutInfo = fillLayoutInfo;

    override var onMouseEntered = function(event: MouseEvent) {
        if (tooltip != null) tooltip.activate();
    }
    override var onMouseExited = function(event: MouseEvent) {
        if (tooltip != null) tooltip.deactivate();
    }
    override var onMousePressed = function(event: MouseEvent) {
        if (tooltip != null) tooltip.deactivate();
    }
}

class CharLabel extends BaseLabel {
    override var font = charLargeFont;
}

class ActionButton extends Button {
    override var font = textFont;
    override var layoutInfo = fillLayoutInfo;
}

class ActionRadioButton extends ActionButton, Toggle {
    override public var selected = false on replace { strong = selected }
}

// Projection mode

def actionProjPers = function(): Void { perspectiveRadio.selected = true; universeFX.setProjectionMode("Perspective") };
def actionProjParal = function(): Void { parallelRadio.selected = true; universeFX.setProjectionMode("Parallel") };

def projectionGroup = ToggleGroup{};

def perspectiveRadio: ActionRadioButton = ActionRadioButton {
    text: "Perspective"
    tooltip: CubeTooltip { text: "Perspective projection mode" }
    toggleGroup: projectionGroup
    action: actionProjPers
    selected: true
}
def parallelRadio: ActionRadioButton = ActionRadioButton {
    text: "Parallel"
    tooltip: CubeTooltip { text: "Parallel projection mode" }
    toggleGroup: projectionGroup
    action: actionProjParal
}

// VantagePoints

def actionVantagepoint = function(vp: String): Void {
    universeFX.resetRotation();
    rotationSlider.value = 50;
    universeFX.setVantagePoint(vp);
}
//
def batsButton: ActionButton = ActionButton {
    font: charFont;
    text: String.valueOf(Character.toChars(10057))
    tooltip: CubeTooltip { text: "Dingbats viewpoint" }
    action: function() { actionVantagepoint("Front") }
};
def arrowsButton: ActionButton = ActionButton {
    font: charFont
    text: String.valueOf(Character.toChars(8646))
    tooltip: CubeTooltip { text: "Arrows viewpoint" }
    action: function() { actionVantagepoint("Back") }
};
def mathButton: ActionButton = ActionButton {
    font: charFont
    text: String.valueOf(Character.toChars(8807))
    tooltip: CubeTooltip { text: "Math Operators viewpoint" }
    action: function() { actionVantagepoint("Left") }
};
def latinButton: ActionButton = ActionButton {
    text: String.valueOf(Character.toChars(76))
    tooltip: CubeTooltip { text: "Latin viewpoint" }
    action: function() { actionVantagepoint("Right") }
};
def homeButton: ActionButton = ActionButton {
    text: "Home"
    tooltip: CubeTooltip { text: "Home viewpoint" }
    action: function() { actionVantagepoint("Home") }
};
def centerButton: ActionButton = ActionButton {
    text: "Center"
    tooltip: CubeTooltip { text: "Center viewpoint" }
    disable: bind not perspectiveRadio.selected;
    action: function() { actionVantagepoint("Center") }
};

// Rotation

def rotationLabel = BaseLabel {
    text: "<  Rotation  >"
    tooltip: CubeTooltip { text: "Rotation direction and speed" }
}
def rotationSlider = Slider {
    blockIncrement: 1
    min: 0
    max: 100
    value: 50       // Initial value set in universe
    tooltip: CubeTooltip { text: "Rotation direction and speed" }
    layoutInfo: fillLayoutInfo
}

// Cube transparency

def cubeTransLabel = BaseLabel {
    text: "Transparency"
    tooltip: CubeTooltip { text: "Cube's transparency" }
}
def cubeTransSlider = Slider {
    blockIncrement: 1
    min: 0
    max: 100
    value: 30     // Initial value set in universe
    tooltip: CubeTooltip { text: "Cube's transparency" }
    layoutInfo: fillLayoutInfo
}

// Effects

var sepiaValue: Number = 1.0;
var bloomValue: Number = 0.5;
var blurValue: Number = 10.0;

var sepiaEnabled = false;
var blurEnabled = false;
var bloomEnabled = false;

def sepiaEffect = SepiaTone {
    level: bind if (sepiaEnabled) effectSlider.value / 100.0 else sepiaValue
}
def blurEffect = GaussianBlur {
    radius: bind if (blurEnabled) effectSlider.value else blurValue
}
def bloomEffect = Bloom {
    threshold: bind if (bloomEnabled) effectSlider.value / 100.0 else bloomValue
}

def effectSliderLabel = BaseLabel {
    text: "- - -"
    tooltip: CubeTooltip { text: "Effect attribute" }
}
def effectSlider = Slider {
    blockIncrement: 1
    value: 0
    disable: true
    layoutInfo: fillLayoutInfo
}

def effectGroup = ToggleGroup{};

def noEffectRadio: ActionRadioButton = ActionRadioButton {
    text: "No Effect"
    tooltip: CubeTooltip { text: "JavaFX effect : None" }
    toggleGroup: effectGroup
    action: function() {
        noEffectRadio.selected = true;

        sepiaValue = sepiaEffect.level;
        bloomValue = bloomEffect.threshold;
        blurValue = blurEffect.radius;
        sepiaEnabled = false;
        blurEnabled = false;
        bloomEnabled = false;

        backgroundPanel.effect = null;
        controlGroup.effect = null;
        fxCanvas3DComp.effect = null;

        effectSlider.value = 0;
        effectSlider.disable = true;
        effectSliderLabel.text = "- - -";
    };
    selected: true
}
def sepiaEffectRadio: ActionRadioButton = ActionRadioButton {
    text: "SepiaTone"
    tooltip: CubeTooltip { text: "JavaFX effect : SepiaTone" }
    toggleGroup: effectGroup
    action: function() {
        sepiaEffectRadio.selected = true;

        sepiaValue = sepiaEffect.level;
        bloomValue = bloomEffect.threshold;
        blurValue = blurEffect.radius;
        blurEnabled = false;
        bloomEnabled = false;

        backgroundPanel.effect = sepiaEffect;
        controlGroup.effect = sepiaEffect;
        fxCanvas3DComp.effect = sepiaEffect;

        effectSlider.max = 100;
        effectSlider.min = 0;
        effectSlider.value =
        sepiaValue * 100 as Integer;
        effectSlider.disable = false;
        effectSliderLabel.text = "Intensity";

        sepiaEnabled = true;
    }
}
def bloomEffectRadio: ActionRadioButton = ActionRadioButton {
    text: "Bloom"
    tooltip: CubeTooltip { text: "JavaFX effect : Bloom" }
    toggleGroup: effectGroup
    action: function() {
        bloomEffectRadio.selected = true;

        sepiaValue = sepiaEffect.level;
        bloomValue = bloomEffect.threshold;
        blurValue = blurEffect.radius;
        sepiaEnabled = false;
        blurEnabled = false;

        backgroundPanel.effect = bloomEffect;
        controlGroup.effect = bloomEffect;
        fxCanvas3DComp.effect = bloomEffect;

        effectSlider.max = 100;
        effectSlider.min = 0;
        effectSlider.value =
        bloomValue * 100 as Integer;
        effectSlider.disable = false;
        effectSliderLabel.text = "Luminosity";

        bloomEnabled = true;
    }
}
def blurEffectRadio: ActionRadioButton = ActionRadioButton {
    text: "G'Blur"
    tooltip: CubeTooltip { text: "JavaFX effect : GaussianBlur" }
    toggleGroup: effectGroup
    action: function() {
        blurEffectRadio.selected = true;

        sepiaValue = sepiaEffect.level;
        bloomValue = bloomEffect.threshold;
        blurValue = blurEffect.radius;
        sepiaEnabled = false;
        bloomEnabled = false;

        backgroundPanel.effect = null;
        controlGroup.effect = null;
        fxCanvas3DComp.effect = blurEffect;

        effectSlider.max = 63;
        effectSlider.min = 0;
        effectSlider.value = blurValue as Integer;
        effectSlider.disable = false;
        effectSliderLabel.text = "Radius";

        blurEnabled = true;
    }
}

//
// Control Box
//

var backgroundPanel: Rectangle;
var controlGroup: Group;

var codePointCharLabel: CharLabel;
var codePointIntLabel: BaseLabel;

var buttonWidth = bind (controlVBox.layoutBounds.width - controlSpace)/2;
var buttonHeight = controlSpace * 3;

var controlVBox: VBox = VBox {
    spacing: controlSpace*4
    content: [
        Tile {
            hgap: controlSpace vgap: 0 columns: 1
            content: [
                BaseLabel { text: "Java 3D meets JavaFX" font: titleFont },
                BaseLabel { text: "CharacterCube" font: titleFont }
            ]
        },
        // Effects
        VBox {
            spacing: controlSpace
            content: [
                VBox {
                    spacing: controlSpace/2
                    content: [ effectSliderLabel, effectSlider ]
                },
                Tile {
                    autoSizeTiles: false
                    tileWidth: bind buttonWidth
                    tileHeight: bind buttonHeight
                    hgap: controlSpace vgap: controlSpace columns: 2 rows: 2
                    content: [ bloomEffectRadio, blurEffectRadio,
                               sepiaEffectRadio, noEffectRadio ]
                }
            ]
        },
        VBox {
            spacing: controlSpace*2
            content: [
                // Rotation
                VBox {
                    spacing: controlSpace/2
                    content: [ rotationLabel, rotationSlider ]
                },
                // Transparency
                VBox {
                    spacing: controlSpace/2
                    content: [ cubeTransLabel, cubeTransSlider ]
                }
            ]
        },
        // VantagePoints
        VBox {
            spacing: controlSpace
            content: [
                Tile {
                    autoSizeTiles: false
                    tileWidth: bind (controlVBox.layoutBounds.width - controlSpace*3)/4
                    tileHeight: bind buttonHeight
                    hgap: controlSpace vgap: 0 columns: 4 rows: 1
                    content: [arrowsButton, batsButton, latinButton, mathButton]
                },
                Tile {
                    autoSizeTiles: false
                    tileWidth: bind buttonWidth
                    tileHeight: bind buttonHeight
                    hgap: controlSpace vgap: 0 columns: 2 rows: 1
                    content: [homeButton, centerButton]
                }
            ]
        },
        // Projection mode
        Tile {
            autoSizeTiles: false
            tileWidth: bind buttonWidth
            tileHeight: bind buttonHeight
            hgap: controlSpace vgap: 0 columns: 2 rows: 1
            content: [parallelRadio, perspectiveRadio]
        }
    ]
}

// Info box
def infoVBox: VBox = VBox {
    spacing: 0
    content: [
         codePointCharLabel = CharLabel {
            text: "\u2704"
            tooltip: CubeTooltip { text: "Unicode character" }
        },
        Tile {
            autoSizeTiles: false
            tileWidth: bind controlVBox.layoutBounds.width/4
            tileHeight: bind controlSpace * 2
            hgap: 0 vgap: 0 columns: 4 rows: 1
            content: [

                // Frames per second
                BaseLabel {
                    text: "F P S : "
                    tooltip: CubeTooltip { text: "Frames per second" }
                },
                BaseLabel {
                    text: bind String.valueOf(fxCanvas3DComp.fpsPaint)
                    tooltip: CubeTooltip { text: "Frames per second" }
                },
                // Selected character, Unicode CodePoint
                BaseLabel {
                    text: "Char : "
                    tooltip: CubeTooltip { text: "Selected character" }
                },
                codePointIntLabel = BaseLabel {
                    text: "9988"
                    tooltip: CubeTooltip { text: "Unicode code point" }
                }
            ]
        }
    ]
}

def popupMenu = PopupMenu {

    def effectRadioGroup = ToggleGroup{}
    var bloomRadio: PopupRadioMenuItem
    var blurRadio: PopupRadioMenuItem
    var sepiaRadio: PopupRadioMenuItem
    var noneRadio: PopupRadioMenuItem

    def projRadioGroup = ToggleGroup{}
    var parRadio: PopupRadioMenuItem
    var perRadio: PopupRadioMenuItem

    var menuItem: MenuItem
    layoutInfo: LayoutInfo {height: bind menuItem.layoutBounds.height * (if (isApplication) 5 else 4) }

    items: [
        PopupSubMenu {
            text: "Select effect"
            items: [
                bloomRadio = PopupRadioMenuItem {
                    text: "Bloom"
                    toggleGroup: effectRadioGroup
                    action: function() {
                        effectGroup.selectedToggle = bloomEffectRadio;
                        bloomEffectRadio.action.invoke();
                    }
                },
                blurRadio = PopupRadioMenuItem {
                    text: "GaussianBlur"
                    toggleGroup: effectRadioGroup
                    action: function() {
                        effectGroup.selectedToggle = blurEffectRadio;
                        blurEffectRadio.action.invoke();
                    }
                },
                sepiaRadio = PopupRadioMenuItem {
                    text: "SepiaTone"
                    toggleGroup: effectRadioGroup
                    action: function() {
                        effectGroup.selectedToggle = sepiaEffectRadio;
                        sepiaEffectRadio.action.invoke();
                    }
                },
                Separator {},
                noneRadio = PopupRadioMenuItem {
                    text: "No Effect"
                    toggleGroup: effectRadioGroup
                    action: function() {
                        effectGroup.selectedToggle = noEffectRadio;
                        noEffectRadio.action.invoke();
                    }
                }
            ]
        },
        Separator {},
        PopupSubMenu {
            text: "Select viewpoint"     
            items: [
                PopupMenuItem { text: "Arrows" action: function() { actionVantagepoint("Back") } },
                PopupMenuItem { text: "Dingbats" action: function() { actionVantagepoint("Front") } },
                PopupMenuItem { text: "Latin" action: function() { actionVantagepoint("Right") } },
                PopupMenuItem { text: "Math Operators" action: function() { actionVantagepoint("Left") } },
                Separator {},
                PopupMenuItem { text: "Home" action: function() { actionVantagepoint("Home") } },
                PopupMenuItem { text: "Center" action: function() { actionVantagepoint("Center") } }
            ]
        },
        Separator {},
        PopupSubMenu {
            text: "Select projection"
            items: [
                parRadio = PopupRadioMenuItem {
                    text: "Parallel"
                    toggleGroup: projRadioGroup
                    action: function() {
                        projectionGroup.selectedToggle = parallelRadio;
                        actionProjParal.invoke();
                    }
                },
                perRadio = PopupRadioMenuItem {
                    text: "Perspective"
                    toggleGroup: projRadioGroup
                    action: function() {
                        projectionGroup.selectedToggle = perspectiveRadio;
                        actionProjPers.invoke();
                    }
                }
            ]
        },
        Separator {},
        menuItem = PopupMenuItem {
            text: "Stop rotation"
            action: function() { rotationSlider.value = 50 }
        }
    ]

    override var onShowing = function() {
        // Effect 
        if (bloomEffectRadio.selected) {
            effectRadioGroup.selectedToggle = bloomRadio
        }
        else if (blurEffectRadio.selected) {
            effectRadioGroup.selectedToggle = blurRadio
        }
        else if (sepiaEffectRadio.selected) {
            effectRadioGroup.selectedToggle = sepiaRadio
        }
        else {
            effectRadioGroup.selectedToggle = noneRadio
        }

        // Projection
        if (perspectiveRadio.selected) {
            projRadioGroup.selectedToggle = perRadio
        }
        else {
            projRadioGroup.selectedToggle = parRadio
        }
    }

    override var onAction = function(item :MenuItem):Void { hide() }

    visible: false
}
// Complete popupMenu
if (isApplication) {
    insert Separator{} into popupMenu.items;
    insert PopupMenuItem {
        text: "Exit application"
        action: function() { stage.close() }
    } into popupMenu.items;
}

//
// Stage / scene
//
stage = Stage {
    title: "InteractiveMesh : FXCharacterCube"
    resizable: true
    visible: false // !!

    onClose: function() { universeFX.closeUniverse() }

    scene: Scene {
        width: sceneHeight * 1.4
        height: sceneHeight
        fill: backgroundColor

        stylesheets: ["{__DIR__}caspian-desktop-charactercube.css"];

        content: [
            // Context menu
            popupMenu,

            // Background node for popup menu
            backgroundPanel = Rectangle {
                width: bind stage.scene.width
                height: bind stage.scene.height
                fill: backgroundColor
                onMousePressed: function (event) {
                    if (popupMenu.showing)
                        popupMenu.hide();
                }
                onMouseClicked: function(event) {
                    if (event.button == MouseButton.SECONDARY) {
                        popupMenu.show(backgroundPanel, event.screenX+5, event.screenY);
                    }
                }
            },

            // Control box
            controlGroup = Group {
                layoutX: border
                layoutY: border
                content: [
                    // LinearGradient background
                    Rectangle {
                        width: bind controlVBox.layoutBounds.width + border
                        height: bind fxCanvas3DComp.layoutBounds.height
                        fill: LinearGradient {
                            startX: 0.0, startY: 0.0, endX: 0.0, endY: 1.0
                            proportional: true
                            stops: [ Stop { offset: 0.0 color: backgroundColor },
                                     Stop { offset: 0.8 color: Color.rgb(0, 153, 255) },
                                     Stop { offset: 1.0 color: backgroundColor } ]
                        }
                    },
                    // Controls
                    Group {
                        layoutX: border/2
                        content:  controlVBox
                    },
                    // Infos
                    Group {
                        layoutX: border/2
                        layoutY: bind fxCanvas3DComp.layoutBounds.height - infoVBox.layoutBounds.height
                        content: infoVBox
                    }
                ]
            },
            // 3D
            Group {
                layoutX: bind 2.2*border + controlVBox.layoutBounds.width
                layoutY: border
                content: fxCanvas3DComp
            }
        ]
    }
}

//
// Start
//
// JavaTaskBase
universeFX.start();
