
package com.interactivemesh.j3d.testspace.jfx.distortstring;

import com.javafx.preview.control.Menu;
import com.javafx.preview.control.MenuItem;
import com.javafx.preview.control.PopupMenu;
import com.javafx.preview.control.RadioMenuItem;

import javafx.animation.Interpolator;
import javafx.animation.Timeline;

import javafx.geometry.Insets;
import javafx.geometry.HPos;
import javafx.geometry.VPos;

import javafx.scene.CustomNode;
import javafx.scene.Group;
import javafx.scene.Scene;

import javafx.scene.control.Label;
import javafx.scene.control.Separator;
import javafx.scene.control.Slider;
import javafx.scene.control.ToggleGroup;
import javafx.scene.control.Tooltip;

import javafx.scene.effect.Reflection;

import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

import javafx.scene.layout.HBox;
import javafx.scene.layout.LayoutInfo;
import javafx.scene.layout.Priority;
import javafx.scene.layout.Tile;

import javafx.scene.paint.Color;

import javafx.scene.shape.Rectangle;

import javafx.scene.text.Font;
import javafx.scene.text.FontWeight;

import javafx.stage.Screen;
import javafx.stage.Stage;

import javafx.util.Math;

/**
 * DistortStringMain.fx
 * 
 * Version: 6.1
 * Date: 2010/09/19
 *
 * Copyright (c) 2009-2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'DistortStringMain.fx'.
 *
 */

// TODO : How to check for application versus applet ??
// API : getArgument("javafx.applet") will return null if not running as an applet
// API : This is an experimental facility, that may be changed in future versions.
var isApplication: Boolean = (FX.getArgument("javafx.applet") == null);

// Design

def screenHeight: Number = Screen.primary.bounds.height;
def screenDPI: Number = Screen.primary.dpi;

var sceneHeight = screenHeight * 0.7;

// screenHeight >= 1200
var textFontSize = 20;
var titleFontSize = 34;
var border = 50;
var lineWidth = 3;

// screenHeight  < 1024
if (screenHeight < 1024) {
    textFontSize = 16;
    titleFontSize = 28;
    border = 30;
}
// 1024 <= screenHeight < 1200
else if (screenHeight < 1200) {
    textFontSize = 18;
    titleFontSize = 30;
    border = 40;
}

// Bug RT-9312, since 1.3.1
// Logical fonts (Dialog, Serif, etc.) overruns, although text doesn't exceed the available space.

def textFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize);
def menuFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize);
def tipFont = Font { size: textFontSize-2};

def textFG = Color.color(0.75, 0.75, 0.75); //Color.WHITE;
def controlSpace = 10;
def vertSpace = 3;

def buttonColor = textFG; Color.WHITE;
def hoverColor = Color.color(1.0, 0.7, 0.0);   // orange;
def pressColor = Color.color(1.0, 0.2, 0.0);   // dark orange;
def selectColor = Color.color(0.0, 1.0, 0.6);   // neon green
def backgroundColor = Color.color(0.0, 0.4, 0.8);


// Frame
var stage: Stage;

//
var isPanelVisible = true;
var isGLSLavailable = false;

def fillLayoutInfo =  LayoutInfo{hfill: true vfill: true hgrow: Priority.ALWAYS};

// Height of scene + reflection
var sceneReflectHeight = bind (stage.scene.height - 3 * border);
// Fraction set by slider
var reflectionFraction = 0.0 on replace {
    def sceneHeight = sceneReflectHeight*(0.5 + 0.5*(1-reflectionFraction));
    (fxCanvas3DComp.effect as Reflection).fraction = (sceneReflectHeight - sceneHeight) / sceneHeight
}

// FXCanvas3D 
def fxCanvas3DComp: FXCanvas3DComp = FXCanvas3DComp {
    
    // Reflection effect
    effect: Reflection {
        // fraction set in 'reflectionFraction'
        topOpacity: bind topOpacitySlider.value / 100.0
        bottomOpacity: bind bottomOpacitySlider.value / 100.0
    }
    // Resizing
    layoutInfo: LayoutInfo {
        width: bind Math.max(10, stage.scene.width - 2 * border);
        height: bind if (sceneReflectHeight > 0) sceneReflectHeight*(0.5 + 0.5*(1-reflectionFraction)) else 10;
    }

    override var onMouseEntered = function(event: MouseEvent) {
        if (not isPanelVisible and event.y > sceneReflectHeight - 5) {
            panelTransitionIn.playFromStart();
        }
        super.onMouseEntered
    };
}

// UniverseFX
def universeFX = DistortStringUniverseFX {
    
    // Callback of AsyncOperation
    initUniverse: function(universe: DistortStringUniverse): Void {
        // Finish FXCanvas3DComp
        fxCanvas3DComp.initFXCanvas3D(universe);
        // Shader supported ?
        isGLSLavailable = universe.isGLSLavailable();
        woodA.disable = not isGLSLavailable;
        woodB.disable = not isGLSLavailable;
        // Initial shape/texture
        universe.setGoldTexture("A");

        // Show frame
        stage.visible = true;

        reflectionFraction = 0.85;

        printSystemProps();
        universe.printJava3DProps();
    }

    distortSpeed: bind distortionSlider.value as Integer
}

//
// Controls
//

// Viewpoints
def actionFrontVp = function() { universeFX.setVantagePoint("Front") };
def actionSideVp = function() { universeFX.setVantagePoint("Side") };

// Distortion

def actionStartDistort = function() { universeFX.startStopDistort(true) };
def actionStopDistort = function() { universeFX.startStopDistort(false) };

def distortStartStopLabel: ToggleLabel = ToggleLabel {
    text: "Distort"
    tooltip: DistortTooltip{ text: "Start/Stop distort animation" }
    offAction: actionStopDistort
    onAction: actionStartDistort
}
distortStartStopLabel.setSelected(true);

def distortionSlider = Slider {
    min: 1
    max: 100
    value: 30
    blockIncrement: 1
    tooltip: DistortTooltip{ text: "Distortion speed" }
}

// Appearance Selection

def appearanceGroup = RadioLabelGroup {
    selectionChanged: function(selectedRadio: RadioLabel) {
        if (selectedRadio.hover) {
            var type = (selectedRadio as TexRadioLabel).type;
            var mode = (selectedRadio as TexRadioLabel).text;
            if (type.equalsIgnoreCase("Gold")) {
                universeFX.setGoldTexture(mode);
            }
            else if (type.equalsIgnoreCase("Wood")) {
                universeFX.setWoodTexture(mode);
            }
        }
    }
}

def goldA: RadioLabel = TexRadioLabel {
    type: "Gold"
    text: "A"
    tooltip: DistortTooltip{ text: "Spherical environment texture" }
}
def goldB: RadioLabel = TexRadioLabel {
    type: "Gold"
    text: "B"
    tooltip: DistortTooltip{ text: "Distorted object linear texture" }
}
def goldC: RadioLabel = TexRadioLabel {
    type: "Gold"
    text: "C"
    tooltip: DistortTooltip{ text: "Object linear texture" }
}

def woodA: RadioLabel = TexRadioLabel {
    type: "Wood"
    text: "A"
    tooltip: DistortTooltip{ text: "Distorted object linear procedural texture shader" }
}
def woodB: RadioLabel = TexRadioLabel {
    type: "Wood"
    text: "B"
    tooltip: DistortTooltip{ text: "Object linear procedural texture shader" }
}

def goldAppear: HBox = HBox {
    hpos: HPos.CENTER
    spacing: controlSpace
    content: [BaseLabel { text: "Gold" }, goldA, goldB, goldC]
}
def woodAppear: HBox = HBox {
    hpos: HPos.CENTER
    spacing: controlSpace
    content: [BaseLabel { text: "Wood" }, woodA, woodB]
    layoutInfo: fillLayoutInfo;
}

appearanceGroup.radioList = [goldA, goldB, goldC, woodA, woodB];
appearanceGroup.setSelection(goldA);
woodA.disable = true;
woodB.disable = true;

// Frames per second

def fpsLabel = BaseLabel {
    text: "F P S : "
    hpos: HPos.CENTER
    tooltip: DistortTooltip{ text: "Frames per second" }
}
def fpsValueLabel = BaseLabel {
    text: bind String.valueOf(fxCanvas3DComp.fpsPaint)
    hpos: HPos.LEFT
    tooltip: DistortTooltip{ text: "Frames per second" }
}
def mspfLabel = BaseLabel {
    text: "M P F : "
    hpos: HPos.CENTER
    tooltip: DistortTooltip{ text: "Milliseconds per frame" }
}
def mspfValueLabel = BaseLabel {
    text: bind String.valueOf( (fxCanvas3DComp.frameDuration + 0.5) as Integer)
    hpos: HPos.LEFT
    tooltip: DistortTooltip{ text: "Milliseconds per frame" }
}

// Reflection effect

def fractionSlider: Slider = Slider {
    min: 0
    max: 100
    value: 85
    blockIncrement: 1
    override var onMouseReleased = function(event: MouseEvent) {
        reflectionFraction = fractionSlider.value / 100.0
    };
    tooltip: DistortTooltip{ text: "Reflection fraction" }
    layoutInfo: fillLayoutInfo;
}
def topOpacitySlider = Slider {
    min: 0
    max: 100
    value: 90
    blockIncrement: 1
    tooltip: DistortTooltip{ text: "Reflection top opacity" }
    layoutInfo: fillLayoutInfo;
}
def bottomOpacitySlider = Slider {
    min: 0
    max: 100
    value: 20
    blockIncrement: 1
    tooltip: DistortTooltip{ text: "Reflection bottom opacity" }
    layoutInfo: fillLayoutInfo;
}

// Control panel
def controlPanel: CustomNode = CustomNode {
    
    var hbox: HBox;

    override protected function create(): Group {
        return Group {
            content: [
                Rectangle {
                    // Transparent backround to avoid conflicts between Slider and SwingComponent
                    // and to remove panel from scene area
                    opacity: 0.0
                    blocksMouse: true
                    width: bind hbox.layoutBounds.width
                    height: bind hbox.layoutBounds.height
                    
                    override var onMouseExited = function(event: MouseEvent) {
                        if (isPanelVisible and event.y > height-1) {
                            panelTransitionOut.playFromStart();
                        }
                    };
                },

                hbox = HBox {
                    padding: Insets { bottom: controlSpace*2 left: 0 right: 0 top: controlSpace/2 }
                    spacing: controlSpace * 3
                    content: [
                        // Viewpoints
                        Tile {
                            hgap: 0 vgap: vertSpace columns: 1
                            content: [
                                ButtonLabel {
                                    text: "Front"
                                    tooltip: DistortTooltip { text: "Front viewpoit" }
                                    action: actionFrontVp
                                },
                                ButtonLabel {
                                    text: "Side"
                                    tooltip: DistortTooltip { text: "Side viewpoint" }
                                    action: actionSideVp
                                }
                            ]
                        },
                        // Distortion
                        Tile {
                            hgap: 0 vgap: vertSpace columns: 1
                            content: [distortStartStopLabel, distortionSlider]
                        },
                        // Reflection
                        Tile {
                            hgap: 10 vgap: vertSpace columns: 3
                            content: [
                                BaseLabel { text: "Fraction" },
                                BaseLabel { text: "Top Opacity" },
                                BaseLabel { text: "Bottom Opacity" },
                                fractionSlider,
                                topOpacitySlider,
                                bottomOpacitySlider
                            ]
                        },
                        // Appearances
                        Tile {
                            hgap: 0 vgap: vertSpace columns: 1
                            content: [goldAppear, woodAppear]
                        }
                    ]
                }
                    
                    
            ]
        }
    }
}

//
def panelTransitionIn = Timeline {
    repeatCount: 1
    keyFrames : [
        at(0ms) {
            controlPanel.visible => true;
            controlPanel.translateY => controlPanel.translateY;
        },
        at(1000ms) {
            controlPanel.translateY => 0 tween Interpolator.LINEAR;
            isPanelVisible => true;
        }
    ]
}
//
def panelTransitionOut = Timeline {
    repeatCount: 1
    keyFrames : [
        at(0ms) {
            controlPanel.translateY => 0;
        },
        at(1000ms) {
            controlPanel.translateY => controlPanel.layoutBounds.height tween Interpolator.LINEAR;
            controlPanel.visible => false;
            isPanelVisible => false;
        }
    ]
}

// Context menu RadioMenuItem

def popupMenu = PopupMenu {

    def appearGroup = ToggleGroup {}
    var apearMenu: PopupSubMenu
    
    var menuItemPanel: MenuItem
    var menuItemDistort: MenuItem
    layoutInfo: LayoutInfo {height: bind menuItemPanel.layoutBounds.height * (if (isApplication) 6 else 4) }

    items: [
        apearMenu = PopupSubMenu {
            text: "Select appearance"
            items: [
                PopupRadioMenuItem { text: "Gold A"  toggleGroup: appearGroup selected: true
                    action: function() { universeFX.setGoldTexture("A"); appearanceGroup.setSelection(goldA) } },
                PopupRadioMenuItem { text: "Gold B" toggleGroup: appearGroup
                    action: function() {universeFX.setGoldTexture("B"); appearanceGroup.setSelection(goldB) } },
                PopupRadioMenuItem { text: "Gold C" toggleGroup: appearGroup
                    action: function() {universeFX.setGoldTexture("C"); appearanceGroup.setSelection(goldC) } },
                Separator {},
                PopupRadioMenuItem { text: "Wood A" toggleGroup: appearGroup
                    action: function() {universeFX.setWoodTexture("A"); appearanceGroup.setSelection(woodA) } },
                PopupRadioMenuItem { text: "Wood B" toggleGroup: appearGroup
                    action: function() {universeFX.setWoodTexture("B"); appearanceGroup.setSelection(woodB) } }
            ]
        },
        Separator {},
        PopupSubMenu {
            text: "Select viewpoint"
            items: [
                PopupMenuItem { text: "Front" action: actionFrontVp },
                PopupMenuItem { text: "Side" action: actionSideVp }
            ]
        },
        Separator {},
        menuItemDistort = PopupMenuItem { },
        Separator {},
        menuItemPanel = PopupMenuItem { }
    ]

    override var onShowing = function() {
        if (isPanelVisible) {
            menuItemPanel.text = "Move out controls";
            menuItemPanel.action = function() { panelTransitionOut.playFromStart() };
        }
        else {
            menuItemPanel.text = "Move in controls";
            menuItemPanel.action = function() { panelTransitionIn.playFromStart() };
        }

        if (distortStartStopLabel.selected) {
            menuItemDistort.text = "Stop distortion";
            menuItemDistort.action = function() {
                distortStartStopLabel.setSelected(false);
                actionStopDistort.invoke();
            }
        }
        else {
            menuItemDistort.text = "Start distortion";
            menuItemDistort.action = function() {
                distortStartStopLabel.setSelected(true);
                actionStartDistort.invoke();
            }
        }
        
        // Appearance items
        if (goldA.selected)      { appearGroup.selectedToggle = (apearMenu.items[0] as PopupRadioMenuItem) }
        else if (goldB.selected) { appearGroup.selectedToggle = (apearMenu.items[1] as PopupRadioMenuItem) }
        else if (goldC.selected) { appearGroup.selectedToggle = (apearMenu.items[2] as PopupRadioMenuItem) }
        else if (woodA.selected) { appearGroup.selectedToggle = (apearMenu.items[4] as PopupRadioMenuItem) }
        else if (woodB.selected) { appearGroup.selectedToggle = (apearMenu.items[5] as PopupRadioMenuItem) }

        if (not isGLSLavailable) {
            apearMenu.items[4].disable = true;
            apearMenu.items[5].disable = true;
        }
    }

    override var onAction = function(item :MenuItem):Void { hide() }

    visible: false
}
// Complete popupMenu
if (isApplication) {
    insert Separator{} into popupMenu.items;
    insert PopupMenuItem {
        text: "Exit application"
        action: function() { stage.close() }
    } into popupMenu.items;
}

// Headline
def headLine = BaseLabel {
    text: "Java 3D meets JavaFX"
    font: Font.font("Amble Cn", FontWeight.REGULAR, titleFontSize)
};

// Frame / Scene

stage = Stage {
    title: "InteractiveMesh : FXDistortString"
    resizable: true
    visible: false // !!
    fullScreen: false;
    onClose: function() {
        universeFX.closeUniverse();
    }
    scene: Scene {
        width: sceneHeight * 1.5
        height: sceneHeight
        fill: backgroundColor

        var popupPanel: Rectangle;

        content: [
            // Context menu
            popupMenu,

            // Background node for popup menu
            popupPanel = Rectangle {
                width: bind stage.scene.width
                height: bind stage.scene.height
                fill: backgroundColor
                onMousePressed: function (event) {
                    if (popupMenu.showing)
                        popupMenu.hide();
                }
                onMouseClicked: function(event) {
                    if (event.button == MouseButton.SECONDARY) {
                        if (isApplication) {
                            popupMenu.show(popupPanel, event.screenX+5, event.screenY);
                        }
                        else { // Applet workaround
                            popupMenu.show(popupPanel, HPos.LEFT, VPos.TOP, event.x, event.y);
                        }
                    }
                }
            },

            // Headline
            Group {
                layoutX: border + controlSpace
                layoutY: bind border - headLine.height * 0.5
                content: headLine
            },

            // Line border, width = 3
            Rectangle {
                layoutX: border - lineWidth
                layoutY: 2 * border - lineWidth
                width: bind fxCanvas3DComp.width + 2*lineWidth
                height: bind sceneReflectHeight + 2*lineWidth
                fill: textFG
            },

            // Reflection background: black
            Rectangle {
                layoutX: border
                layoutY: 2 * border
                width: bind fxCanvas3DComp.width
                height: bind sceneReflectHeight
                fill: Color.color(0, 0, 0)
            },

            // 3D
            Group {
                layoutX: border
                layoutY: 2 * border
                content: [
                    // FXCanvas3D of Java 3D scene
                    fxCanvas3DComp,
                    // Radio controls
                    Group {
                        translateX: bind (fxCanvas3DComp.width - controlPanel.layoutBounds.width) / 2
                        translateY: bind sceneReflectHeight - controlPanel.layoutBounds.height +5
                        content: controlPanel
                    }
                ]
            },

            // Covers control panel
            Rectangle {
                layoutX: 0
                layoutY: bind stage.scene.height - border + lineWidth
                width: bind stage.scene.width
                height: border - lineWidth
                fill: backgroundColor
            },

            // Frames per second
            Group {
                layoutX: border + controlSpace
                layoutY: bind stage.scene.height - border + fpsLabel.layoutBounds.height * 0.5
                content: Tile {
                    hgap: 0 vgap: 0 columns: 5 // 4 doesn't work
                    nodeHPos: HPos.LEFT
                    content: [fpsLabel, fpsValueLabel, mspfLabel, mspfValueLabel]
                }
            }
        ]
    }
}

//
// Start
//
// JavaTaskBase
universeFX.start();

//
// Control classes
//

class PopupSubMenu extends Menu {
    override var font = menuFont
}
class PopupMenuItem extends MenuItem {
    override var font = menuFont
}
class PopupRadioMenuItem extends RadioMenuItem {
    override var font = menuFont
}

class DistortTooltip extends Tooltip {
    override var font = tipFont;
}

class BaseLabel extends Label {
    override var font = textFont;
    override var textFill = buttonColor;
    override var hpos = HPos.CENTER;
    override var layoutInfo = fillLayoutInfo;

    var selected: Boolean = false;  
    
    override var onMouseEntered = function(event: MouseEvent) {
        if (tooltip != null) tooltip.activate();
    }
    override var onMouseExited = function(event: MouseEvent) {
        if (tooltip != null) tooltip.deactivate();
    }
    override var onMousePressed = function(event: MouseEvent) {
        if (tooltip != null) tooltip.deactivate();
    }
}

class ButtonLabel extends BaseLabel {

    var action: function(): Void;

    override var onMouseEntered = function(event: MouseEvent) {
        if (not pressed) {
            textFill = hoverColor;
        }        
        tooltip.activate();
    };
    override var onMouseExited = function(event: MouseEvent) {
        if (not pressed) {
            textFill = buttonColor;
        }
        tooltip.deactivate();
    };
    override var onMousePressed = function(event: MouseEvent) {
        textFill = pressColor;
        tooltip.deactivate();
    };
    override var onMouseReleased = function(event: MouseEvent) {
        if (disabled)
            return;

        if (hover) {
            action();
            textFill = hoverColor;
        }
        else {
            textFill = buttonColor;
        }
    };
}
class ToggleLabel extends BaseLabel {

    var offAction: function(): Void;
    var onAction: function(): Void;

    function setSelected(select: Boolean) {
        selected = select;
        if (select) {
            textFill = selectColor;
        }
        else {
            textFill = buttonColor;
        }
    }

    override var onMouseEntered = function(event: MouseEvent) {
        if (not pressed) {
            textFill = hoverColor;
        }
        tooltip.activate();
    };
    override var onMouseExited = function(event: MouseEvent) {
        if (not pressed) {
            if (selected) {
                textFill = selectColor;
            }
	    else  {
                textFill = buttonColor;
            }
        }
        tooltip.deactivate();
    };
    override var onMousePressed = function(event: MouseEvent) {
        pressed = true;
        textFill = pressColor;
        tooltip.deactivate();
    };
    override var onMouseReleased = function(event: MouseEvent) {
        pressed = false;
        if (contains(event.x, event.y)) {
            selected = not selected;
            if (selected) {
                onAction();
            }
            else {
                offAction();
            }
        }

    	if (selected) {
            textFill = selectColor;
        }
    	else {
            textFill = buttonColor;
        }
    }
}

class RadioLabel extends BaseLabel {

    var group: RadioLabelGroup;

    var selectAction: function(): Void;

    override var onMouseEntered = function(event: MouseEvent) {
        if (not pressed and not selected) {
            textFill = hoverColor;
        }
        tooltip.activate();
    };
    override var onMouseExited = function(event: MouseEvent) {
        if (not pressed and not selected) {
            textFill = buttonColor;
        }
        tooltip.deactivate();
    };
    override var onMousePressed = function(event: MouseEvent) {
        if (not selected) {
            pressed = true;
            textFill = pressColor;
        }
        tooltip.deactivate(); 
    };
    override var onMouseReleased = function(event: MouseEvent) {
        if (selected or disabled)
            return;
        pressed = false;
        if (hover) {
            group.select(this);
        }
        else {
            textFill = buttonColor;
        }
    };
}

class TexRadioLabel extends RadioLabel {
    var type: String = ""
}

class RadioLabelGroup {

    var radioList: RadioLabel[] on replace {
        for (radio in radioList) {
            radio.group = this;
        }
    };

    var selectionChanged: function(radio: RadioLabel);

    function setSelection(selectedRadio: RadioLabel) {
        for (radio in radioList) {
            radio.selected = false;
            radio.textFill = buttonColor;
        }

        selectedRadio.selected = true;
        selectedRadio.textFill = selectColor;
    }

    function select(selectedRadio: RadioLabel) {
        setSelection(selectedRadio);
        selectionChanged(selectedRadio);
    }

}

// System properties
function printSystemProps() {
    FX.println("");
    FX.println("------------------------------------------------------------------------");
    FX.println("SYSTEM  PROPERTIES");
    FX.println("------------------------------------------------------------------------");
    FX.println("OS      Name     =  {FX.getProperty("javafx.os.name")}");
    FX.println("OS      Arch     =  {FX.getProperty("javafx.os.arch")}");
    FX.println("OS      Version  =  {FX.getProperty("javafx.os.version")}");
    FX.println("------------------------------------------------------------------------");
    FX.println("JavaFX  Version  =  {FX.getProperty("javafx.version")}");
    FX.println("Java    Version  =  {FX.getProperty("javafx.java.version")}");
    FX.println("Java    Vender   =  {FX.getProperty("javafx.java.vendor")}");
    FX.println("------------------------------------------------------------------------");
}
