
package com.interactivemesh.j3d.testspace.jfx.tuxanaglyph;

import com.javafx.preview.control.Menu;
import com.javafx.preview.control.MenuItem;
import com.javafx.preview.control.PopupMenu;

import javafx.geometry.HPos;

import javafx.scene.Group;
import javafx.scene.Scene;

import javafx.scene.control.Button;
import javafx.scene.control.Label;
import javafx.scene.control.Separator;
import javafx.scene.control.Slider;
import javafx.scene.control.Toggle;
import javafx.scene.control.ToggleGroup;
import javafx.scene.control.Tooltip;

import javafx.scene.input.MouseButton;
import javafx.scene.input.MouseEvent;

import javafx.scene.layout.HBox;
import javafx.scene.layout.LayoutInfo;
import javafx.scene.layout.Tile;

import javafx.scene.paint.Color;

import javafx.scene.shape.Rectangle;

import javafx.scene.text.Font;
import javafx.scene.text.FontWeight;

import javafx.stage.Screen;
import javafx.stage.Stage;

import javafx.util.Math;

/**
 * TuxAnaglyphMain.fx
 *
 * Version: 3.0
 * Date: 2010/09/20
 *
 * Copyright (c) 2009-2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'TuxAnaglyphMain.fx'.
 *
 */

// TODO : How to check for application versus applet ??
// API : getArgument("javafx.applet") will return null if not running as an applet 
// API : This is an experimental facility, that may be changed in future versions. 
var isApplication: Boolean = (FX.getArgument("javafx.applet") == null);

// Design

def backgroundColor = Color.color(0.0, 0.4, 0.8); // #0066cc
def textColor = Color.color(0.8, 0.8, 0.8); // #cccccc

def screenHeight: Number = Screen.primary.bounds.height;

// screenHeight >= 1200
var textFontSize = 18;
var titleFontSize = 34;
var sceneHeight = 900;
var border = 50;
var lineWidth = 3;

// screenHeight  < 1024
if (screenHeight < 1024) {
    textFontSize = 14;
    titleFontSize = 26;
    border = 30;
    sceneHeight = 700;
}
// 1024 <= screenHeight < 1200
else if (screenHeight < 1200) {
    textFontSize = 16;
    titleFontSize = 30;
    border = 40;
    sceneHeight = 800;
}

// Bug RT-9312, since 1.3.1
// Logical fonts (Dialog, Serif, etc.) overruns, although text doesn't exceed the available space.

def titleFont = Font.font("Amble Cn", FontWeight.REGULAR, titleFontSize);
def textFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize);
def menuFont = Font.font("Amble Cn", FontWeight.BOLD, textFontSize);
def tipFont = Font { size: textFontSize};

def controlSpace = 10;
def gapH = 4;
def gapV = 5;


var isAnaglyph: Boolean = true;

// Frame
var stage: Stage;

// FXCanvas3D
def fxCanvas3DAnaglyphComp = FXCancas3DAnaglyphComp {
    // Resizing
    layoutInfo: LayoutInfo {
        width: bind Math.max(10, stage.scene.width - 2 * border);
        height: bind Math.max(10, stage.scene.height - 4 * border);
    }

    updateFPS: function(fps: Integer, mpf: Integer): Void {
        fpsValueLabel.text = String.valueOf(fps);
        mpfValueLabel.text = String.valueOf(mpf);
    }
}
def fxCanvas3DMonoComp = FXCancas3DMonoComp {
    // Resizing
    layoutInfo: LayoutInfo {
        width: bind Math.max(10, stage.scene.width - 2 * border);
        height: bind Math.max(10, stage.scene.height - 4 * border);
    }
    
    updateFPS: function(fps: Integer, mpf: Integer): Void {
        fpsValueLabel.text = String.valueOf(fps);
        mpfValueLabel.text = String.valueOf(mpf);
    }
}

var fxCanvas3DGroup: Group = Group {content: fxCanvas3DAnaglyphComp};

// UniverseFX
def universeFX = TuxAnaglyphUniverseFX {
    // Callback of AsyncOperation
    initUniverse: function(universe: TuxAnaglyphUniverse): Void {
        // Finish FXCanvas3DComps
        fxCanvas3DAnaglyphComp.initFXCanvas3D(universe);
        fxCanvas3DMonoComp.initFXCanvas3D(universe);
        // setup navigators for both FXCanvas3DComps
        universe.setupNavigators();

        // Show frame
        stage.visible = true;
    }
    // Change rotation speed
    rotationValue: bind rotationSlider.value as Integer
    // Interpupillary distance
    eyesPosition: bind distanceSlider.value as Integer
}

//
// Controls
//

class PopupSubMenu extends Menu {
    override var font = menuFont
}
class PopupMenuItem extends MenuItem {
    override var font = menuFont
}

class TuxTooltip extends Tooltip {
    override var font = tipFont;
}

class BaseLabel extends Label {
    init {
        styleClass = "tux-label";
        
        hpos = HPos.CENTER;
    }
    override var font = textFont;

    override var onMouseEntered = function(event: MouseEvent) {
        if (tooltip != null) tooltip.activate();
    }
    override var onMouseExited = function(event: MouseEvent) {
        if (tooltip != null) tooltip.deactivate();
    }
    override var onMousePressed = function(event: MouseEvent) {
        if (tooltip != null) tooltip.deactivate();
    }
}

class ActionButton extends Button {
    init {
        font = textFont;
        layoutInfo = LayoutInfo { hfill: true vfill: true };
    }
}

class ActionRadioButton extends ActionButton, Toggle {
    override public var selected = false on replace { strong = selected }
}

// View mode

def viewTGroup = ToggleGroup{};

def monoscopicRadio: ActionRadioButton = ActionRadioButton {
    text: "Monoscopic"
    tooltip: TuxTooltip { text: "Monoscopic view mode" }
    toggleGroup: viewTGroup
    action: function(): Void {
        isAnaglyph = false;
        universeFX.viewMode = "Mono";
        fxCanvas3DGroup.content = fxCanvas3DMonoComp;
        distanceSlider.disable = true;
        monoscopicRadio.selected = true;
    }
}
def stereoscopicRadio: ActionRadioButton = ActionRadioButton {
    text: "Stereoscopic"
    tooltip: TuxTooltip { text: "Stereoscopic view mode" }
    toggleGroup: viewTGroup
    action: function(): Void {
        isAnaglyph = true;
        universeFX.viewMode = "Stereo";
        fxCanvas3DGroup.content = fxCanvas3DAnaglyphComp;
        distanceSlider.disable = false;
        stereoscopicRadio.selected = true;
    }
    selected: true
}

// CubeDimension

def dimTGroup = ToggleGroup{};

class DimRadioButton extends ActionRadioButton {
    var dim: Integer;
    def tuxTriangles = 13744;
    init {
        toggleGroup = dimTGroup;
        text = dim.toString();
        def dim3 = dim*dim*dim;
        def trias = tuxTriangles*dim3;
        action = function() {
            universeFX.cubeDimension = dim;
            selected = true;
        }
        tooltip = if (dim == 1) TuxTooltip { text: "1 Tux,  {%,d trias} triangles" }
                  else TuxTooltip { text: "{dim3} Tuxes,  {%,d trias} triangles" };
    }
}

def dimRadioButtons: DimRadioButton[] = [
    DimRadioButton { dim: 1 }, DimRadioButton { dim: 2 selected: true }
    DimRadioButton { dim: 3 }, DimRadioButton { dim: 4 },
    DimRadioButton { dim: 5 }, DimRadioButton { dim: 6 },
    DimRadioButton { dim: 7 }, DimRadioButton { dim: 8 }
];

// VantagePoints

def actionVpFront = function(): Void { universeFX.setVantagePoint("Front") };
def actionVpCenter = function(): Void { universeFX.setVantagePoint("Center") };
def actionVpTop = function(): Void { universeFX.setVantagePoint("Top") };
def actionVpBottom = function(): Void { universeFX.setVantagePoint("Bot") };

def frontTButton = ActionButton {
    text: "Front"
    tooltip: TuxTooltip { text: "Front viewpoint" }
    action: actionVpFront
};
def centerTButton = ActionButton {
    text: "Center"
    tooltip: TuxTooltip { text: "Center viewpoint" }
    action: actionVpCenter
};
def topTButton = ActionButton {
    text: "Top"
    tooltip: TuxTooltip { text: "Top viewpoint" }
    action: actionVpTop
};
def botTButton = ActionButton {
    text: "Bottom"
    tooltip: TuxTooltip { text: "Bottom viewpoint" }
    action: actionVpBottom
};


// Rotation

def actionResetRotation = function(): Void {
        universeFX.resetRotation();
        rotationSlider.value = 50;
};
def rotationLabel = ActionButton {
    text: "    <   Rotation   >    "
    tooltip: TuxTooltip { text: "Reset rotation" }
    action: actionResetRotation
    layoutInfo: LayoutInfo { hfill: true vfill: true }
};
def rotationSlider = Slider {
    min: 0
    max: 100
    value: 50       // Initial value set in universe
    blockIncrement: 1
    tooltip: TuxTooltip { text: "Direction and speed" }
    layoutInfo: LayoutInfo { hfill: true vfill: true }
}

// Interpupillary Distance

def distanceLabel = ActionButton {
    text: bind "Distance : {String.valueOf(distanceSlider.value as Integer)} mm"
    tooltip: TuxTooltip { text: "Reset distance" }
    action: function(): Void {
        distanceSlider.value = 66;
    }
    layoutInfo: LayoutInfo { hfill: true vfill: true }
}
def distanceSlider = Slider {
    min: 0
    max: 100
    value: 66     // Initial value set in universe
    blockIncrement: 1
    tooltip: TuxTooltip { text: "Interpupillary distance [0, 100] mm" }
    layoutInfo: LayoutInfo { hfill: true vfill: true }
}

//
// Control Box
//
var controlGroup: Group = Group {
    
    var controlBox: HBox
    
    content: [
        // Transparent background to avoid conflict between Slider and SwingComponent
        Rectangle {
            //fill: Color.BLUE
            opacity: 0.0
            blocksMouse: true
            width: bind controlBox.layoutBounds.width
            height: bind controlBox.layoutBounds.height
        },
        controlBox = HBox {
            spacing: controlSpace * 2
            content: [
                // VantagePoints
                Tile {
                    hgap: gapH vgap: gapV columns: 2
                    content: [frontTButton, topTButton, centerTButton, botTButton]
                },
                // CubeDimension
                Tile {
                    hgap: gapH vgap: gapV columns: 4
                    content: dimRadioButtons
                },
                // Rotation, Distance
                Tile {
                    hgap: controlSpace*2 vgap: gapV columns: 2
                    content: [rotationLabel, distanceLabel, rotationSlider, distanceSlider]
                },
                // View mode
                Tile {
                    hgap: gapH vgap: gapV columns: 1
                    content: [monoscopicRadio, stereoscopicRadio]
                }
            ]
        }
    ]
}

def popupMenu = PopupMenu {

    var menuItem: MenuItem
    layoutInfo: LayoutInfo {height: bind menuItem.layoutBounds.height * (if (isApplication) 4 else 3) }

    items: [
        PopupSubMenu {
            text: "Select viewpoint"
            items: [
                PopupMenuItem { text: "Front" action: actionVpFront },
                PopupMenuItem { text: "Top" action: actionVpTop },
                PopupMenuItem { text: "Bottom" action: actionVpBottom },
                PopupMenuItem { text: "Center" action: actionVpCenter }
            ]
        },
        Separator {},
        menuItem = PopupMenuItem {
            text: "Reset rotation"
            action: actionResetRotation
        }
    ]

    override var onShowing = function() {
    }

    override var onAction = function(item :MenuItem):Void { hide() }

    visible: false
}
// Complete popupMenu
if (isApplication) {
    insert Separator{} into popupMenu.items;
    insert PopupMenuItem {
        text: "Exit application"
        action: function() { stage.close() }
    } into popupMenu.items;
}

// Headline
def headLine = BaseLabel {
    text: "Java 3D meets JavaFX"
    font: titleFont
};

// Frames per second
def fpsLabel = BaseLabel {
    text: "F P S"
    tooltip: TuxTooltip { text: "Frames per second" }
}
def fpsValueLabel = BaseLabel {
    text: "0"
}
// Millisecond per frame
def mpfLabel = BaseLabel {
    text: "M P F"
    tooltip: TuxTooltip { text: "Millisec per frame" }
}
def mpfValueLabel = BaseLabel {
    text: "0"
}

//
// Stage / scene
//
stage = Stage {
    title: "InteractiveMesh : FXTuxInStereo - Anaglyphs"
    resizable: true
    visible: false // !!
    onClose: function() {
        universeFX.closeUniverse();
    }
    scene: Scene {
        width: sceneHeight * 1.4
        height: sceneHeight
        fill: backgroundColor

        stylesheets: ["{__DIR__}caspian-desktop-tuxanaglyph.css"];

        var popupPanel: Rectangle;
        
        content: [
            // Context menu
            popupMenu,

            // Background node for popup menu
            popupPanel = Rectangle {
                width: bind stage.scene.width
                height: bind stage.scene.height
                fill: backgroundColor
                onMousePressed: function (event) {
                    if (popupMenu.showing)
                        popupMenu.hide();
                }
                onMouseClicked: function(event) {
                    if (event.button == MouseButton.SECONDARY) {
                        popupMenu.show(popupPanel, event.screenX+5, event.screenY);
                    }
                }
            },

            // Headline
            Group {
                layoutX: border + controlSpace
                layoutY: bind border - headLine.height * 0.5
                content: headLine
            },
            // FPS value
            Group {
                var tile: Tile;
                layoutX: bind stage.scene.width - border - fpsLabel.layoutBounds.width*2 - controlSpace*2
                layoutY: bind border - tile.layoutBounds.height * 0.5
                content: tile = Tile {
                    hgap: controlSpace*2 vgap: gapV columns: 2
                    content: [fpsLabel, fpsValueLabel, mpfLabel, mpfValueLabel]
                }
            }
            
            // Line border, width = lineWidth
            Rectangle {
                layoutX: border - lineWidth;
                layoutY: 2 * border - lineWidth;
                width: bind if (isAnaglyph) (fxCanvas3DAnaglyphComp.width + 2*lineWidth)
                                            else (fxCanvas3DMonoComp.width + 2*lineWidth)
                height: bind if (isAnaglyph) (fxCanvas3DAnaglyphComp.height + 2*lineWidth)
                                            else (fxCanvas3DMonoComp.height + 2*lineWidth)
                fill: textColor
            },

            // FXCanvas3D of Java 3D scene
            Group {
                layoutX: border
                layoutY: 2 * border
                content: fxCanvas3DGroup
            },
            
            // Controls
            Group {
                layoutX: bind (stage.scene.width - controlGroup.layoutBounds.width) / 2
                layoutY: bind stage.scene.height + lineWidth - border - controlGroup.layoutBounds.height*0.5
                content: controlGroup
            }          
        ]
    }
}

//
// Start
//
// JavaTaskBase
universeFX.start();
