
package com.interactivemesh.j3d.testspace.jfx.tuxanaglyph;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GradientPaint;
import java.awt.Graphics2D;

import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseEvent;

import java.awt.image.BufferedImage;

import java.util.Enumeration;
import java.util.HashMap;

import javax.media.j3d.Alpha;
import javax.media.j3d.Appearance;
import javax.media.j3d.Background;
import javax.media.j3d.Behavior;
import javax.media.j3d.BoundingBox;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.Bounds;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.GeometryArray;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.ImageComponent2D;
import javax.media.j3d.Locale;
import javax.media.j3d.Material;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.PickInfo;
import javax.media.j3d.RotationInterpolator;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.VirtualUniverse;
import javax.media.j3d.WakeupOnBehaviorPost;

import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Vector3d;

import com.sun.j3d.utils.pickfast.PickCanvas;

// X3DModelImporter
import com.interactivemesh.j3d.interchange.ext3d.XImportException;
import com.interactivemesh.j3d.interchange.ext3d.XShapeReader;

// FXCanvas3DAnaglyph API 1.0, see http://www.interactivemesh.org/testspace/j3dmeetsjfx.html
import com.interactivemesh.j3d.community.gui.FXCanvas3DAnaglyph;

// FXCanvas3D API 3.0, see http://www.interactivemesh.org/testspace/j3dmeetsjfx.html
import com.interactivemesh.j3d.community.gui.FXCanvas3DSB;
import com.interactivemesh.j3d.community.gui.FXCanvas3DRepainter;

// OrbitBehaviorAnaglyph 0.5
import com.interactivemesh.j3d.community.utils.navigation.orbitanaglyph.OrbitBehaviorAnaglyph;
// OrbitBehaviorInterim 2.1, see http://www.interactivemesh.org/testspace/orbitbehavior.html
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim;

// JavaFX
import javafx.async.RunnableFuture;

/**
 * TuxAnaglyphUniverse.java
 *
 * Version: 3.0
 * Date: 2010/09/20
 *
 * Copyright (c) 2009-2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'TuxAnaglyphUniverse.java'.
 *
 */
final class TuxAnaglyphUniverse implements RunnableFuture {

    static {
        System.out.println("FXTuxAnaglyph : Copyright (c) 2009-2010 August Lammersdorf, www.InteractiveMesh.com.");
    }

    private BoundingSphere      globalBounds        =   null;

    private Locale              locale              =   null;
    private View                viewLeft            =   null;
    private View                viewRight           =   null;
    private View                viewMono            =   null;
    private PhysicalBody        physicalBodyLeft    =   null;
    private PhysicalBody        physicalBodyRight   =   null;
    private ViewPlatform        viewsPlatform       =   null;

    private FXCanvas3DAnaglyph  fxCanvas3DAnaglyph  =   null;
    private FXCanvas3DSB        fxCanvas3DMono      =   null;
    private ComponentAdapter    resizeListener      =   null;
    private MouseListener       mousePickAdapter    =   null;

    private BranchGroup         sceneBranch         =   null;
    private BranchGroup         viewBranch          =   null;
    private BranchGroup         enviBranch          =   null;

    private Background          background          =   null;

    private TransformGroup      positionGroup       =   null;
    private TransformGroup      rotationGroup       =   null;
    private BranchGroup         tuxBranch           =   null;

    private RotationInterpolator rotInterp          =   null;
    private Alpha               rotationAlpha       =   null;
    private long                rotationStartTime   =   0;

    private OrbitBehaviorAnaglyph orbitBehAnaglyph  =   null;
    private OrbitBehaviorInterim orbitBehMono       =   null;

    private PickCanvas          pickCanvasAnaglyph  = 	null;
    private PickCanvas          pickCanvasMono      = 	null;

    private BoundingBox         currBoundingBox     =   new BoundingBox();
    private double              currSceneRadius     =   1.0f;
    private int                 currWidth           =   0;
    private int                 currHeight          =   0;

    private boolean             isStereo            =   true;
    private int                 currCubeDim         =   2;

    private Color               bgColor0            =   new Color(0, 102, 204); // RGB 0.0f, 0.4f, 0.8f, HSB 209, 100,  80
    private Color               bgColor1            =   new Color(0, 153, 255); // RGB 0.0f, 0.6f, 1.0f, HSB 203, 100, 100

    private SpeedBehavior       speedBehavior       =   null;
    private VantagePointBehavior vpExecutor         =   null;

    private HashMap<String, VantagePoint> vantagepointHM = new HashMap<String, VantagePoint>();

    // Tux: shared geometries and appearances

    private GeometryArray   bodyGeom        =   null;
    private GeometryArray   frontGeom       =   null;
    private GeometryArray   feetGeom        =   null;
    private GeometryArray   eyesGeom        =   null;
    private GeometryArray   pupilsGeom      =   null;
    private GeometryArray   mouthGeom       =   null;

    private Appearance      frontAppear     =   null;
    private Appearance      feetAppear      =   null;
    private Appearance      eyesAppear      =   null;
    private Appearance      pupilsAppear    =   null;
    private Appearance      mouthAppear     =   null;


    TuxAnaglyphUniverse() {
    }

    // JavaFX Interface RunnableFuture
    @Override
    public void run() {
        initUniverse();
    }

    // Creates and returns the anaglyphic lightweight 3D canvas
    FXCanvas3DAnaglyph createFXCanvas3DAnaglyph(FXCanvas3DRepainter repainter, JPanel parent) {

        try {
            GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();

            fxCanvas3DAnaglyph = new FXCanvas3DAnaglyph(gCT);
            fxCanvas3DAnaglyph.setRepainter(repainter);
        }
        catch (Exception e) {
            System.out.println("TuxAnaglyphUniverse: FXCanvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }

        // Optional: less 'flickering' during resizing
        fxCanvas3DAnaglyph.setBackground(bgColor0);

        // Due to Java 3D's implementation of off-screen rendering:
        // 1. Set size
        // 2. Provide a parent
        // 3. Get the heavyweight off-screen Canvas3D and add this to the view object

        // 1. Size
        Dimension dim = parent.getSize();
        if (dim.width < 1 || dim.height < 1) {
            dim.width = 100;
            dim.height = 100;
            parent.setSize(dim);
        }
        parent.setPreferredSize(dim);
        fxCanvas3DAnaglyph.setPreferredSize(dim);
        fxCanvas3DAnaglyph.setSize(dim);

        // 2. Parent
        // BorderLayout
        parent.setLayout(new BorderLayout());
        parent.add(fxCanvas3DAnaglyph, BorderLayout.CENTER);

        // 3. Heavyweight off-screen Canvas3Ds of the lightweight FXCanvas3DAnaglyph
        Canvas3D offCanvas3DLeft = fxCanvas3DAnaglyph.getOffscreenCanvas3DLeft();
        Canvas3D offCanvas3DRight = fxCanvas3DAnaglyph.getOffscreenCanvas3DRight();
        if (offCanvas3DLeft != null) {
            // View renders into the off-screen Canvas3D
            viewLeft.addCanvas3D(offCanvas3DLeft);
            viewRight.addCanvas3D(offCanvas3DRight);

            // PickCanvas operates on the not visible off-screen Canvas3D
    	    pickCanvasAnaglyph = new PickCanvas(offCanvas3DRight, sceneBranch);
    	    pickCanvasAnaglyph.setMode(PickInfo.PICK_GEOMETRY);
    	    pickCanvasAnaglyph.setFlags(PickInfo.NODE);
    	    pickCanvasAnaglyph.setTolerance(4.0f);
        }
        else {
            System.out.println("TuxAnaglyphUniverse: OffscreenCanvas3D Stereo = null !!");
            System.exit(0);
        }

        fxCanvas3DAnaglyph.addComponentListener(resizeListener);
        fxCanvas3DAnaglyph.addMouseListener(mousePickAdapter);

        orbitBehAnaglyph.setAWTComponent(fxCanvas3DAnaglyph);

        return fxCanvas3DAnaglyph;
    }

    // Creates and returns the monoglyphic lightweight 3D canvas
    FXCanvas3DSB createFXCanvas3DMono(FXCanvas3DRepainter repainter, JPanel parent) {

        try {
            GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();

            fxCanvas3DMono = new FXCanvas3DSB(gCT);
            fxCanvas3DMono.setRepainter(repainter);
        }
        catch (Exception e) {
            System.out.println("TuxAnaglyphUniverse: FXCanvas3D failed !!");
            e.printStackTrace();
            System.exit(0);
        }

        // Optional: less 'flickering' during resizing
        fxCanvas3DMono.setBackground(bgColor0);

        // Due to Java 3D's implementation of off-screen rendering:
        // 1. Set size
        // 2. Provide a parent
        // 3. Get the heavyweight off-screen Canvas3D and add this to the view object

        // 1. Size
        Dimension dim = parent.getSize();
        if (dim.width < 1 || dim.height < 1) {
            dim.width = 100;
            dim.height = 100;
            parent.setSize(dim);
        }
        parent.setPreferredSize(dim);
        fxCanvas3DMono.setPreferredSize(dim);
        fxCanvas3DMono.setSize(dim);

        // 2. Parent
        // BorderLayout
        parent.setLayout(new BorderLayout());
        parent.add(fxCanvas3DMono, BorderLayout.CENTER);

        // 3. Heavyweight off-screen Canvas3D of the lightweight FXCanvas3D
        Canvas3D offCanvas3DMono = fxCanvas3DMono.getOffscreenCanvas3D();
        if (offCanvas3DMono != null) {
            // View renders into the off-screen Canvas3D
            viewMono.addCanvas3D(offCanvas3DMono);

            // PickCanvas operates on the not visible off-screen Canvas3D
    	    pickCanvasMono = new PickCanvas(offCanvas3DMono, sceneBranch);
    	    pickCanvasMono.setMode(PickInfo.PICK_GEOMETRY);
    	    pickCanvasMono.setFlags(PickInfo.NODE);
    	    pickCanvasMono.setTolerance(4.0f);
        }
        else {
            System.out.println("TuxAnaglyphUniverse: OffscreenCanvas3D Mono = null !!");
            System.exit(0);
        }

        fxCanvas3DMono.addMouseListener(mousePickAdapter);

        orbitBehMono.setAWTComponent(fxCanvas3DMono);

        return fxCanvas3DMono;
    }

    // Stage.onClose
    void closeUniverse() {
        /*
        viewLeft.removeAllCanvas3Ds();
        viewRight.removeAllCanvas3Ds();
        */
        viewLeft.attachViewPlatform(null);
        viewRight.attachViewPlatform(null);
        locale.getVirtualUniverse().removeAllLocales();
    }

    //
    // Universe interaction
    //

    // Mono <--> Stereo
    void setViewMode(String mode) {

        Point3d p3d = new Point3d();
        Transform3D t3d = new Transform3D();
            
        if (mode.equalsIgnoreCase("Mono") && isStereo) {
            isStereo = false;

            viewLeft.attachViewPlatform(null);
            viewRight.attachViewPlatform(null);
            viewMono.attachViewPlatform(viewsPlatform);

            orbitBehAnaglyph.getRotationCenter(p3d);
            orbitBehAnaglyph.getViewingTransform(t3d);
            orbitBehAnaglyph.setEnable(false);

            orbitBehMono.setRotationCenter(p3d);
            orbitBehMono.setEnable(true);
            orbitBehMono.setViewingTransform(t3d);

            fxCanvas3DAnaglyph.removeComponentListener(resizeListener);
            fxCanvas3DMono.addComponentListener(resizeListener);
        }
        else if (mode.equalsIgnoreCase("Stereo") && !isStereo) {
            isStereo = true;

            viewMono.attachViewPlatform(null);
            viewLeft.attachViewPlatform(viewsPlatform);
            viewRight.attachViewPlatform(viewsPlatform);

            orbitBehMono.getRotationCenter(p3d);
            orbitBehMono.getViewingTransform(t3d);
            orbitBehMono.setEnable(false);

            orbitBehAnaglyph.setRotationCenter(p3d);
            orbitBehAnaglyph.setEnable(true);
            orbitBehAnaglyph.setViewingTransform(t3d);

            fxCanvas3DMono.removeComponentListener(resizeListener);
            fxCanvas3DAnaglyph.addComponentListener(resizeListener);
        }
    }

    void setVantagePoint(String name) {
        VantagePoint vp = vantagepointHM.get(name);
        vpExecutor.setVP(vp);
        vpExecutor.postId(VantagePointBehavior.APPLY_VP);
    }

    void setCubeDimension(int dim) {
        if (currCubeDim == dim)
            return;
        currCubeDim = dim;
        tuxBranch.detach();
        createTuxBranch(currCubeDim);
 //setVantagePoint("Front");
        setupNavigators();
    }

    // [0, 100] Interpupillary Distance
    void setEyesPosition(int value) {

        double dist = value/(2.0*1000.0);

        physicalBodyLeft.setRightEyePosition(new Point3d(dist, 0.0, 0.0));
        physicalBodyLeft.setLeftEyePosition(new Point3d(-dist, 0.0, 0.0));
        physicalBodyRight.setRightEyePosition(new Point3d(dist, 0.0, 0.0));
        physicalBodyRight.setLeftEyePosition(new Point3d(-dist, 0.0, 0.0));
    }

    void resetRotation() {
        rotationAlpha.pause(rotationStartTime);
    }

    // Rotation speed and direction
    // Update via a Behavior provides smoother animation
    void setRotationSpeed(int speed) {
        speedBehavior.setSpeed(speed);
    }

    //
    // Create universe
    //
    private void initUniverse() {

        createUniverse();
        createScene();

        createVantagePoints();

        loadTux();

        createTuxBranch(currCubeDim);

        setLive();

        setVantagePoint("Front");

        // Common MouseListener for picking
        mousePickAdapter = new MouseListener() {
            @Override
            public void mouseClicked(MouseEvent event) {
                int clicks = event.getClickCount();
                if (clicks == 2 && SwingUtilities.isLeftMouseButton(event)) {
                    changeRotationCenter(event);
                }
            }
            public void mouseEntered(MouseEvent e) {}
            public void mouseExited(MouseEvent e) {}
            public void mousePressed(MouseEvent e) {}
            public void mouseReleased(MouseEvent e) {}
        };
    }

    void setupNavigators() {
 
        BoundingSphere sphereBounds = new BoundingSphere(currBoundingBox);

        currSceneRadius = sphereBounds.getRadius(); // Needed for VantagePoints

        orbitBehAnaglyph.setTransFactors(currSceneRadius/4.0f, currSceneRadius/4.0f);
        orbitBehAnaglyph.setZoomFactor(currSceneRadius/4.0f);
        orbitBehAnaglyph.setRotFactors(0.4f, 0.4f);
        orbitBehAnaglyph.setClippingBounds(sphereBounds);

        orbitBehMono.setTransFactors(currSceneRadius/4.0f, currSceneRadius/4.0f);
        orbitBehMono.setZoomFactor(currSceneRadius/4.0f);
        orbitBehMono.setRotFactors(0.4f, 0.4f);
        orbitBehMono.setClippingBounds(sphereBounds);
    }

    private void createScene() {

        // Tux parent

        positionGroup = new TransformGroup();
        positionGroup.setCapability(TransformGroup.ALLOW_CHILDREN_EXTEND);
        positionGroup.setCapability(TransformGroup.ALLOW_CHILDREN_WRITE);
        positionGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        // Rotation

        rotationGroup = new TransformGroup();
        rotationGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        rotationGroup.addChild(positionGroup);

        rotationAlpha = new Alpha();
        rotationAlpha.setLoopCount(-1);
        // Initialize rotation
        rotationAlpha.setIncreasingAlphaDuration(1500 * 47);    // start speed = min speed
        rotationStartTime = System.currentTimeMillis();         // start time
        rotationAlpha.setStartTime(rotationStartTime);          // start
        rotationAlpha.pause(rotationStartTime);                 // pause at start time

        rotInterp = new RotationInterpolator(rotationAlpha, rotationGroup);
        rotInterp.setSchedulingBounds(globalBounds);

        sceneBranch.addChild(rotInterp);
        sceneBranch.addChild(rotationGroup);

    }

    private void createTuxBranch(int dim) {

        // Tux : 13.744 triangles, 10 Shape3Ds

        // dim 1 :   1 * 13.744 =    13,744
        // dim 2 :   8 * 13.744 =   109,952
        // dim 3 :  27 * 13.744 =   371,088
        // dim 4 :  64 * 13.744 =   879,616
        // dim 5 : 125 * 13.744 = 1,718,000
        // dim 6 : 216 * 13.744 = 2,968,704
        // dim 7 : 343 * 13.744 = 4,714,192
        // dim 8 : 512 * 13.744 = 7,036,928

        // Tux BoundingBox
        Point3d lower = new Point3d(-0.81812, 0.01011, -0.31617);
        Point3d upper = new Point3d(0.81812, 1.45735, 0.28825);

        int xDist = 2;
        int yDist = 2;
        int zDist = 2;

        double halfX = ( (upper.x - lower.x) + (dim-1)*xDist ) / 2;
        double halfY = ( (upper.y - lower.y) + (dim-1)*yDist ) / 2;
        double halfZ = ( (upper.z - lower.z) + (dim-1)*zDist ) / 2;

        Point3d center = new Point3d(lower.x + halfX, lower.y + halfY, lower.z + halfZ);

        // Tux BoundingBox centered at (0, 0, 0)
        Point3d lowerBox = new Point3d(-halfX, -halfY, -halfZ);
        Point3d upperBox = new Point3d(halfX, halfY, halfZ);
        currBoundingBox.setLower(lowerBox);
        currBoundingBox.setUpper(upperBox);

        // Center Tux at (0, 0, 0)
        Transform3D rotT3D = new Transform3D();
        rotT3D.setTranslation(new Vector3d(-center.x, -center.y, -center.z));
        positionGroup.setTransform(rotT3D);

        // Tux' branch
        tuxBranch = new BranchGroup();
        tuxBranch.setCapability(BranchGroup.ALLOW_DETACH);

        TransformGroup tuxTG = null;
        Transform3D tuxT3D = new Transform3D();

        float xTrans = 0;
        float yTrans = 0;
        float zTrans = 0;

        for (int i=0; i < dim; i++) {               // z axis

            // Shared Appearances for bodies
            Appearance[] appears = createAppearances(dim, 1);   
            int t = appears.length-1;

            for (int j=0; j < dim; j++) {           // y axis
                for (int k=0; k < dim; k++) {       // x axis

                    tuxTG = new TransformGroup();
                    tuxT3D.setTranslation(new Vector3d(xTrans, yTrans, zTrans));
                    tuxTG.setTransform(tuxT3D);

                    tuxTG.addChild(new Shape3D(bodyGeom, appears[t])); 
                    t--;
                    tuxTG.addChild(new Shape3D(frontGeom, frontAppear)); 
                    tuxTG.addChild(new Shape3D(feetGeom, feetAppear));
                    tuxTG.addChild(new Shape3D(eyesGeom, eyesAppear));
                    tuxTG.addChild(new Shape3D(pupilsGeom, pupilsAppear));
                    tuxTG.addChild(new Shape3D(mouthGeom, mouthAppear));

                    tuxBranch.addChild(tuxTG);

                    xTrans += xDist;
                }                                   // x axis

                xTrans = 0;
                yTrans += yDist;
            }                                       // y axis

            yTrans = 0;
            zTrans += zDist;

            t = appears.length-1;
        }                                           // z axis

        positionGroup.addChild(tuxBranch);
    }

    // Appearances for body and front
    private Appearance[] createAppearances(int dim, float bFactor) {

        Appearance[] appears = new Appearance[dim*dim];

        int k = 0;
        int direction = 1;
        float h = 0;

        float step = 1.0f / (dim*dim);

        for (int i=0; i < dim; i++) {
            for (int j=0; j < dim; j++) {

                appears[k] = new Appearance();

                float hAnaglyph = h;  //h; //0.5f + 0.16f*h; //0.3333f + 0.3333f*h;
                appears[k].setMaterial(getMaterial(hAnaglyph, bFactor));

                h += step * direction;
                k++;
            }

            direction *= (-1);
            if (direction < 0)
                h += step * (dim-1);
            else
                h += step * (dim+1);
        }

        return appears;
    }

    private Material getMaterial(float hue, float bFactor) {
        Material mat = new Material();
        mat.setAmbientColor(0.0f, 0.0f, 0.0f);
        mat.setDiffuseColor(new Color3f(Color.getHSBColor(hue, 1.0f, 0.8f*bFactor))); // 0.85 0.7
        mat.setEmissiveColor(new Color3f(Color.getHSBColor(hue, 1.0f, 0.2f*bFactor)));
        mat.setSpecularColor(0.2f, 0.2f, 0.2f);
        mat.setShininess(16f);
        return mat;
    }

    private void loadTux() {
        
        // TuxModel-AllShapesNoTongue.x3d -> file only includes named Shape nodes
        XShapeReader x3dImporter = new XShapeReader();
        try {
            x3dImporter.read(this.getClass().getResource("resources/TuxModel-AllShapesNoTongue.x3d"));
        }
        catch (XImportException e) {
            e.printStackTrace();
            return;
        }

        // Typesafe retrieving of shapes (no check if null)
        Shape3D[] x3dShapes = x3dImporter.getImport();
        
        HashMap<String, Shape3D> namedShapes = new HashMap<String, Shape3D>();
        for (Shape3D shape : x3dShapes) {
            namedShapes.put(shape.getName(), shape);
        }
         
        // Alternative :
        // Map<String, SceneGraphObject> namedSGOs = x3dImporter.getNamedSGOs();   
        // Shape3D crankarm = (Shape3D)namedSGOs.get("TuxBody")
        // etc.

        x3dImporter.close();

        Shape3D shape3D = namedShapes.get("TuxBody");
        bodyGeom = (GeometryArray)shape3D.getGeometry();

        shape3D = namedShapes.get("TuxEyes");
        eyesGeom = (GeometryArray)shape3D.getGeometry();
        eyesAppear = shape3D.getAppearance();

        shape3D = namedShapes.get("TuxFeet");
        feetGeom = (GeometryArray)shape3D.getGeometry();
        feetAppear = shape3D.getAppearance();

        shape3D = namedShapes.get("TuxFront");
        frontGeom = (GeometryArray)shape3D.getGeometry();
        frontAppear = shape3D.getAppearance();
        
        shape3D = namedShapes.get("TuxMouth");
        mouthGeom = (GeometryArray)shape3D.getGeometry();
        mouthAppear = shape3D.getAppearance();

        shape3D = namedShapes.get("TuxPupils");
        pupilsGeom = (GeometryArray)shape3D.getGeometry();
        pupilsAppear = shape3D.getAppearance();
   }

    private void changeRotationCenter(MouseEvent event) {

        PickInfo pickInfo = null;

        if (isStereo) {
            pickCanvasAnaglyph.setShapeLocation(event.getX(), event.getY());
            pickInfo = pickCanvasAnaglyph.pickClosest();
        }
        else {
            pickCanvasMono.setShapeLocation(event.getX(), event.getY());
            pickInfo = pickCanvasMono.pickClosest();
        }

        if (pickInfo != null) {

            Shape3D pickedShape3D = (Shape3D)pickInfo.getNode();

            Transform3D locToVWord = new Transform3D();
            pickedShape3D.getLocalToVworld(locToVWord);

            Point3d rotationPoint = new Point3d();

            Bounds bounds = pickedShape3D.getBounds();

            if (bounds instanceof BoundingBox) {
                BoundingBox pickedBox = (BoundingBox)bounds;

                Point3d lower = new Point3d();
                Point3d upper = new Point3d();
                pickedBox.getLower(lower);
                pickedBox.getUpper(upper);

                rotationPoint.set(lower.x + (upper.x - lower.x)/2,
                                  lower.y + (upper.y - lower.y)/2,
                                  lower.z + (upper.z - lower.z)/2);
            }
            else {
                BoundingSphere pickedSphere = null;
                if (bounds instanceof BoundingSphere)
                    pickedSphere = (BoundingSphere)bounds;
                else
                    pickedSphere = new BoundingSphere(bounds);

                pickedSphere.getCenter(rotationPoint);
            }

            locToVWord.transform(rotationPoint);

            if (isStereo)
                orbitBehAnaglyph.setRotationCenter(rotationPoint, true); // isLookAtRotCenter
            else
                orbitBehMono.setRotationCenter(rotationPoint, true); // isLookAtRotCenter
        }
    }

    // Set live
    private void setLive() {
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
    }

    // Base world
    private void createUniverse() {

        // Bounds
        globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);

        //
        // Viewing
        //

        // Individual objects for left and right views due to interactive update in 'setEyeDistance(int value)'
        physicalBodyLeft = new PhysicalBody();
        physicalBodyRight = new PhysicalBody();

        PhysicalEnvironment pE = new PhysicalEnvironment();

        viewLeft = new View();
        viewLeft.setPhysicalBody(physicalBodyLeft);
        viewLeft.setPhysicalEnvironment(pE);

        viewRight = new View();
        viewRight.setPhysicalBody(physicalBodyRight);
        viewRight.setPhysicalEnvironment(pE);

        viewMono = new View();
        viewMono.setPhysicalBody(new PhysicalBody());
        viewMono.setPhysicalEnvironment(pE);

        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);

        //
        // BranchGraphs
        //
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();

        // ViewBranch

        TransformGroup viewTG = new TransformGroup();
        viewTG.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        viewsPlatform = new ViewPlatform();
        viewLeft.attachViewPlatform(viewsPlatform);
        viewRight.attachViewPlatform(viewsPlatform);

        orbitBehAnaglyph = new OrbitBehaviorAnaglyph(OrbitBehaviorAnaglyph.REVERSE_ALL);
        orbitBehAnaglyph.setViewingTransformGroup(viewTG);
        orbitBehAnaglyph.setVpViews(viewLeft, viewRight);
        orbitBehAnaglyph.setSchedulingBounds(globalBounds);
        orbitBehAnaglyph.setClippingEnabled(true);
        orbitBehAnaglyph.setProjectionMode(View.PERSPECTIVE_PROJECTION);

        orbitBehMono = new OrbitBehaviorInterim(OrbitBehaviorInterim.REVERSE_ALL);
        orbitBehMono.setViewingTransformGroup(viewTG);
        orbitBehMono.setVpView(viewMono);
        orbitBehMono.setSchedulingBounds(globalBounds);
        orbitBehMono.setClippingEnabled(true);
        orbitBehMono.setProjectionMode(View.PERSPECTIVE_PROJECTION);
        orbitBehMono.setEnable(false);

        speedBehavior = new SpeedBehavior();
        speedBehavior.setSchedulingBounds(globalBounds);

        vpExecutor = new VantagePointBehavior(orbitBehAnaglyph, orbitBehMono);
        vpExecutor.setSchedulingBounds(globalBounds);

        DirectionalLight headLight = new DirectionalLight();
        headLight.setInfluencingBounds(globalBounds);

        viewTG.addChild(viewsPlatform);
        viewTG.addChild(orbitBehAnaglyph);
        viewTG.addChild(orbitBehMono);
        viewTG.addChild(speedBehavior);
        viewTG.addChild(vpExecutor);
        viewTG.addChild(headLight);

        viewBranch.addChild(viewTG);

        // EnviBranch

        background = new Background();
        background.setCapability(Background.ALLOW_IMAGE_WRITE);
        background.setApplicationBounds(globalBounds);
        background.setColor(new Color3f(bgColor0));
        // Background image
        resizeListener = new ComponentAdapter() {
            @Override
            public void componentResized(ComponentEvent e) {
                updateBackgroundImage(e.getComponent().getWidth(), e.getComponent().getHeight());
            }
        };

        enviBranch.addChild(background);
    }

    private void updateBackgroundImage(int width, int height) {

        currWidth = width;
        currHeight = height;

        BufferedImage gradientImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
        Graphics2D g2d = (Graphics2D)gradientImage.getGraphics();

        int splitHeight = (int)(height/5f); // yUp !

        GradientPaint gp = new GradientPaint(0, 0,           bgColor0,
                                             0, splitHeight, bgColor1);
        g2d.setPaint(gp);
        g2d.fillRect(0, 0, width, splitHeight);

        gp = new GradientPaint(0, splitHeight, bgColor1,
                               0, height,      bgColor0);
        g2d.setPaint(gp);
        g2d.fillRect(0, splitHeight, width, height);

        g2d.dispose();

        background.setImage(new ImageComponent2D(ImageComponent2D.FORMAT_RGB, gradientImage, true, true));
    }

    private void createVantagePoints() {

        vantagepointHM.clear();

        double fov = Math.toRadians(45); // default value is used

        // VantagePoints
        new VantagePoint("Front",  new Point3d(0, 0, 10), new Point3d(), new Vector3d(0.0, 1.0, 0.0), new Point3d(), fov);
        new VantagePoint("Center", new Point3d(0, 0, 0), new Point3d(0, 0, -1), new Vector3d(0.0, 1.0, 0.0), new Point3d(), fov);
        new VantagePoint("Top",    new Point3d(0, 10, 0), new Point3d(), new Vector3d(0.0, 0.0,-1.0), new Point3d(), fov);
        new VantagePoint("Bot",    new Point3d(0, -10, 0), new Point3d(), new Vector3d(0.0, 0.0, 1.0), new Point3d(), fov);
    }

    private double getDistToSceneCenter(double fov) {
        // Extra space
        double borderFactor = 1.0;
        // Consider ratio of canvas' width and height
        double ratioFactor = 1.0;
        int c3dWidth = currWidth;
        int c3dHeight = currHeight;
        if (c3dWidth > c3dHeight)
            ratioFactor = (double)c3dWidth/c3dHeight;  
            
        double distToSceneCenter = ratioFactor * borderFactor * currSceneRadius / Math.tan(fov/2);
        
        return distToSceneCenter;
    }

    private final class VantagePoint {

        private String      name        =   null;

        private Point3d     eyePerspect =   new Point3d();
        private Point3d     viewCenter  =   new Point3d();
        private Vector3d    up          =   new Vector3d();
        private Point3d     rotCenter   =   new Point3d();
        private double      fov         =   Math.PI/4;

        private VantagePoint(String name, Point3d eyePer, Point3d viewCenter,
                             Vector3d up, Point3d rotationCenter, double fov) {

            this.name = name;

            this.eyePerspect.set(eyePer);
            this.viewCenter.set(viewCenter);
            this.up.set(up);
            this.rotCenter.set(rotationCenter);
            this.fov = fov;

            vantagepointHM.put(name, this);
        }

        @Override
        public String toString() {
            return name;
        }

        private void applyTo(OrbitBehaviorAnaglyph navigator0,
                             OrbitBehaviorInterim  navigator1) {
            
            double dist = getDistToSceneCenter(fov);
            
            if (name.equalsIgnoreCase("Front")) {
                eyePerspect.z = dist;
            }
            else if (name.equalsIgnoreCase("Top")) {
                eyePerspect.y = dist;
            }
            else if (name.equalsIgnoreCase("Bot")) {
                eyePerspect.y = -dist;
            }

            navigator0.setViewingTransform(eyePerspect, viewCenter, up, rotCenter);
            navigator1.setViewingTransform(eyePerspect, viewCenter, up, rotCenter);
            //navigator.setFieldOfView(fov);
        }
    }

    // Sets vantage point in behavior scheduler
    private final class VantagePointBehavior extends Behavior {

        private static final int        APPLY_VP = 1;
        private WakeupOnBehaviorPost    applyPost = new WakeupOnBehaviorPost(this, APPLY_VP);

        private OrbitBehaviorAnaglyph   orbitBehA = null;
        private OrbitBehaviorInterim    orbitBehM = null;
        private VantagePoint            vantagePoint = null;

        private VantagePointBehavior(OrbitBehaviorAnaglyph orbitBeh0,
                                     OrbitBehaviorInterim  orbitBeh1) {
            orbitBehA = orbitBeh0;
            orbitBehM = orbitBeh1;
        }

        private void setVP(VantagePoint vp) {
            vantagePoint = vp;
        }

        @Override
        public void initialize() {
            wakeupOn(applyPost);
        }
        @Override
        public void processStimulus(Enumeration criteria) {

            if (vantagePoint != null) {
                vantagePoint.applyTo(orbitBehA, orbitBehM);
            }

            vantagePoint = null;

            wakeupOn(applyPost);
        }
    }

    private final class SpeedBehavior extends Behavior {

        private WakeupOnBehaviorPost    updatePost  =   new WakeupOnBehaviorPost(this, 0);
        private int                     postID      =   Integer.MIN_VALUE;

        private SpeedBehavior() {
        }

        private void setSpeed(int speed) {
            postId(speed);
        }
        
        @Override
        public void initialize() {
            wakeupOn(updatePost);
        }
        @Override
        public void processStimulus(Enumeration criteria) {

            while (criteria.hasMoreElements()) {
                postID = ((WakeupOnBehaviorPost)criteria.nextElement()).getTriggeringPostId();
            }
            // The last one
            updateRotationSpeed(postID);

            wakeupOn(updatePost);
        }
        
        private void updateRotationSpeed(int speed) {
            if (rotationAlpha == null)
                return;

            if (speed > 99)
                speed = 99;
            else if (speed < 1)
                speed = 1;

            // Loop duration determines rotation speed
            long duration = 0;
            long pauseTime = 0;
            float pauseValue = 0;

            if (speed > 52) {

                // isPaused
                if (rotationAlpha.isPaused()) {

                    if (rotationAlpha.getMode() == Alpha.INCREASING_ENABLE) {
                        rotationAlpha.resume();
                        return;
                    }

                    pauseTime = rotationAlpha.getPauseTime();
                }
                // Pause
                else {
                    pauseTime = System.currentTimeMillis();
                    rotationAlpha.pause(pauseTime);
                }

                pauseValue = rotationAlpha.value();

                if (rotationAlpha.getMode() != Alpha.INCREASING_ENABLE)
                    rotationAlpha.setMode(Alpha.INCREASING_ENABLE);

                // New IncreasingAlphaDuration
    //          duration = (long)1000 *((100 - speed));                 // [47, 1] non linear
                duration = (long)(1500 * ( ( 47.0/(speed - 52) ) ));    // [47, 1] linear

                // Offset according to rotationAlpha's pauseValue and the new IncreasingAlphaDuration
                long resumeOffsetTime = (long)(pauseValue * duration);

                rotationAlpha.setIncreasingAlphaDuration(duration);

                // Resume

                // Alpha source code: newStartTime = oldStartTime + resumeTime - pauseTime
                // Start immediately and adapt new duration:
                //   => System.currentTimeMillis() - resumeOffsetTime = oldStartTime + resumeTime - pauseTime
                //   => resumeTime = System.currentTimeMillis() - resumeOffsetTime - oldStartTime + pauseTime

                long resumeTime = System.currentTimeMillis() - resumeOffsetTime
                                  - rotationAlpha.getStartTime() + pauseTime;

                rotationAlpha.resume(resumeTime);
            }
            else if (speed < 48) {

                // isPaused
                if (rotationAlpha.isPaused()) {

                    if (rotationAlpha.getMode() == Alpha.DECREASING_ENABLE) {
                        rotationAlpha.resume();
                        return;
                    }

                    pauseTime = rotationAlpha.getPauseTime();
                }
                // Pause
                else {
                    pauseTime = System.currentTimeMillis();
                    rotationAlpha.pause(pauseTime);
                }

                pauseValue = rotationAlpha.value();

                if (rotationAlpha.getMode() != Alpha.DECREASING_ENABLE)
                    rotationAlpha.setMode(Alpha.DECREASING_ENABLE);

                // New DecreasingAlphaDuration
    //          duration = (long)(1000 * speed);                        // [47, 1] non linear
                duration = (long)(1500 * ( ( 47.0/(48 - speed) ) ));    // [47, 1] linear

                // Offset according to pauseValue and the new DecreasingAlphaDuration
                long resumeOffsetTime = (long)((1-pauseValue) * duration);

                rotationAlpha.setDecreasingAlphaDuration(duration);

                // Resume
                rotationAlpha.resume(System.currentTimeMillis() - resumeOffsetTime
                                     - rotationAlpha.getStartTime() + pauseTime);
            }
            else {
                if (!rotationAlpha.isPaused()) {
                    rotationAlpha.pause(System.currentTimeMillis());
                }
            }
        }
        
    }
}
