/*
 * j3d-examples: src.resources.images.bg.jpg
 *
 * Copyright (c) 2007 Sun Microsystems, Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * - Redistribution of source code must retain the above copyright
 *   notice, this list of conditions and the following disclaimer.
 *
 * - Redistribution in binary form must reproduce the above copyright
 *   notice, this list of conditions and the following disclaimer in
 *   the documentation and/or other materials provided with the
 *   distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of
 * contributors may be used to endorse or promote products derived
 * from this software without specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any
 * kind. ALL EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND
 * WARRANTIES, INCLUDING ANY IMPLIED WARRANTY OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON-INFRINGEMENT, ARE HEREBY
 * EXCLUDED. SUN MICROSYSTEMS, INC. ("SUN") AND ITS LICENSORS SHALL
 * NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF
 * USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR
 * ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL,
 * CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND
 * REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF OR
 * INABILITY TO USE THIS SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or
 * intended for use in the design, construction, operation or
 * maintenance of any nuclear facility.
 *
 */
package com.interactivemesh.j3d.testspace.jfx.canvascube;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Image;

import java.awt.image.BufferedImage;

import java.io.FileNotFoundException;
import java.io.IOException;

import java.net.URL;

import java.util.Enumeration;

import javax.imageio.ImageIO;

import javax.media.j3d.Alpha;
import javax.media.j3d.Appearance;
import javax.media.j3d.Background;
import javax.media.j3d.Behavior;
import javax.media.j3d.BoundingSphere;
import javax.media.j3d.BranchGroup;
import javax.media.j3d.Canvas3D;
import javax.media.j3d.DirectionalLight;
import javax.media.j3d.GeometryArray;
import javax.media.j3d.GraphicsConfigTemplate3D;
import javax.media.j3d.Group;
import javax.media.j3d.ImageComponent2D;
import javax.media.j3d.Locale;
import javax.media.j3d.Material;
import javax.media.j3d.PhysicalBody;
import javax.media.j3d.PhysicalEnvironment;
import javax.media.j3d.PointLight;
import javax.media.j3d.PolygonAttributes;
import javax.media.j3d.PositionPathInterpolator;
import javax.media.j3d.RotationInterpolator;
import javax.media.j3d.Shape3D;
import javax.media.j3d.Texture2D;
import javax.media.j3d.TextureAttributes;
import javax.media.j3d.Transform3D;
import javax.media.j3d.TransformGroup;
import javax.media.j3d.View;
import javax.media.j3d.ViewPlatform;
import javax.media.j3d.ViewSpecificGroup;
import javax.media.j3d.VirtualUniverse;
import javax.media.j3d.WakeupOnBehaviorPost;

import javax.swing.JPanel;

import javax.vecmath.Color3f;
import javax.vecmath.Point3d;
import javax.vecmath.Point3f;
import javax.vecmath.Vector3d;

// X3DModelImporter
import com.interactivemesh.j3d.interchange.ext3d.XImportException;
import com.interactivemesh.j3d.interchange.ext3d.XShapeReader;

// FXCanvas3DMV API
import com.interactivemesh.j3d.community.gui.FXCanvas3DMV;
import com.interactivemesh.j3d.community.gui.FXCanvas3DMVControl;

// JavaFX
import javafx.async.RunnableFuture;

/**
 * TuxInTheBoxUniverse.java
 *
 * Version: 4.0
 * Date: 2010/09/20
 *
 * Copyright (c) 2009-2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'TuxInTheBoxUniverse.java'.
 *
 * Image bg512.jpg: See license notice above: j3d-examples: src.resources.images.bg.jpg
 *
 * Images DukeNyaNya512.jpg, SwingingDuke512.png: Released under the
 * Berkeley Software Distribution (BSD) License: https://duke.dev.java.net/
 */
final class TuxInTheBoxUniverse implements RunnableFuture {

    static {
        System.out.println("FXTuxInTheBox : Copyright (c) 2009-2010 August Lammersdorf, www.InteractiveMesh.com.");
    }

    private BoundingSphere              globalBounds        =   null;

    private Locale                      locale              =   null;

    private BranchGroup                 sceneBranch         =   null;
    private BranchGroup                 viewBranch          =   null;
    private BranchGroup                 enviBranch          =   null;

    private View[]                      views               =   null;
    private TransformGroup[]            viewTGs             =   null;
    
    private Appearance                  faceAppear          =   null;
    private Texture2D[]                 faceTextures        =   null;
    private BufferedImage[]             faceIconImages      =   null;

    private RotationInterpolator        rotInterp           =   null;
    private PositionPathInterpolator    zPosInterpolator    =   null;
    private PositionPathInterpolator    yPosInterpolator    =   null;

    private FieldOfViewBehavior         fovBehavior         =   null;
    private volatile boolean            isFovActive         =   false;

    private FXCanvas3DMVControl         fxC3DMVControl      =   null;
    private FXCanvas3DMVSquare[]        fxCanvas3DMVs       =   null;

    private Color                       bgColor = new Color(0.0f, 0.4f, 0.8f);


    TuxInTheBoxUniverse() {
    }

    // JavaFX Interface RunnableFuture
    @Override
    public void run() {
        initUniverse();
    }

    // Multi view FXCanvas3D
    FXCanvas3DMV getFXCanvas3DMV(int num) {
        return fxCanvas3DMVs[num];
    }

    // Multi view control
    FXCanvas3DMVControl getFXC3dMVControl() {
        return fxC3DMVControl;
    }

    BufferedImage[] getIconImages() {
        return faceIconImages;
    }
    
    void setBoxFaceTexture(int index) {
        if (faceTextures == null)
            return;
        if (index >= 0 && index < faceTextures.length && faceTextures[index] != null)
            faceAppear.setTexture(faceTextures[index]);
    }

    //
    // Scene interaction
    //
    void setViewing(int fov) {
        // Update requires two frames !
        if (isFovActive) {
            return;
        }

        if (fov < 10) {
            fov = 10;
        } 
        else if (fov > 120) {
            fov = 120;
        }

        if (fovBehavior != null) {
            isFovActive = true;
            fovBehavior.fov = Math.toRadians(fov);
            fovBehavior.postId(FieldOfViewBehavior.UPDATE);
        }
    }

    void closeUniverse() {

        rotInterp.setEnable(false);
        zPosInterpolator.setEnable(false);
        yPosInterpolator.setEnable(false);

//      locale.getVirtualUniverse().removeAllLocales(); /* if active: application doesn't terminate correctly */

        for (View view : views) {
            view.removeAllCanvas3Ds();
            view.attachViewPlatform(null);
        }
    }

    private void initUniverse() {

        createUniverse();
        createScene();
        setLive();

        createFXCanvas3DMV();
    }

    private void createScene() {

        // Tux BoundingBox, center: 0, 0.73373, -0.01396
        //Point3d lower = new Point3d(-0.81812, 0.01011, -0.31617);
        //Point3d upper = new Point3d(0.81812, 1.45735, 0.28825);

        long startTime = System.currentTimeMillis();

        TransformGroup rotationGroup = new TransformGroup();
        rotationGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        Alpha rotationAlpha = new Alpha();
        rotationAlpha.setMode(Alpha.DECREASING_ENABLE | Alpha.INCREASING_ENABLE);
        rotationAlpha.setLoopCount(-1);
        rotationAlpha.setDecreasingAlphaDuration(5000);
        rotationAlpha.setIncreasingAlphaDuration(5000);
        rotationAlpha.setDecreasingAlphaRampDuration(500);
        rotationAlpha.setIncreasingAlphaRampDuration(500);
        rotationAlpha.setStartTime(startTime);

        rotInterp = new RotationInterpolator(rotationAlpha, rotationGroup);
        rotInterp.setSchedulingBounds(globalBounds);

        TransformGroup yPositionGroup = new TransformGroup();
        yPositionGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        TransformGroup zPositionGroup = new TransformGroup();
        zPositionGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);

        Alpha yPositionAlpha = new Alpha();
        yPositionAlpha.setMode(Alpha.DECREASING_ENABLE | Alpha.INCREASING_ENABLE);
        yPositionAlpha.setLoopCount(-1);
        yPositionAlpha.setDecreasingAlphaDuration(3000);
        yPositionAlpha.setIncreasingAlphaDuration(3000);
        yPositionAlpha.setDecreasingAlphaRampDuration(300);
        yPositionAlpha.setIncreasingAlphaRampDuration(300);
        yPositionAlpha.setStartTime(startTime);

        Alpha zPositionAlpha = new Alpha();
        zPositionAlpha.setMode(Alpha.DECREASING_ENABLE | Alpha.INCREASING_ENABLE);
        zPositionAlpha.setLoopCount(-1);
        zPositionAlpha.setDecreasingAlphaDuration(2000);
        zPositionAlpha.setIncreasingAlphaDuration(2000);
        zPositionAlpha.setDecreasingAlphaRampDuration(200);
        zPositionAlpha.setIncreasingAlphaRampDuration(200);
        zPositionAlpha.setStartTime(startTime);

        float[] knots = {0.0f, 1.0f};
        Point3f[] yPos = {new Point3f(0, -1.04f, 0), new Point3f(0, -0.45f, 0)};
        Point3f[] zPos = {new Point3f(0, 0, 0), new Point3f(0, 0, 0.6f)};

        zPosInterpolator = new PositionPathInterpolator(zPositionAlpha, zPositionGroup,
                                                        new Transform3D(), knots, zPos);
        zPosInterpolator.setSchedulingBounds(globalBounds);

        yPosInterpolator = new PositionPathInterpolator(yPositionAlpha, yPositionGroup,
                                                        new Transform3D(), knots, yPos);
        yPosInterpolator.setSchedulingBounds(globalBounds);

        zPositionGroup.addChild(yPositionGroup);
        rotationGroup.addChild(zPositionGroup);

        // Tux

        try {
            // TuxModel-AllShapesNoTongue.x3d -> file only includes named Shape nodes
            XShapeReader x3dImporter = new XShapeReader();

            x3dImporter.read(this.getClass().getResource("resources/TuxModel-AllShapesNoTongue.x3d"));

            Shape3D[] x3dShapes = x3dImporter.getImport();
        
            for (Shape3D shape : x3dShapes) {
                yPositionGroup.addChild(shape);
            }
        
            x3dImporter.close();
        }
        catch (XImportException e) {
            e.printStackTrace();
            return;
        }

        // Cube
        
        GeometryArray[] cubeFaces = CubeFaces.create(GeometryArray.COORDINATES |
                                                    GeometryArray.NORMALS |
                                                    //GeometryArray.COLOR_3 |
                                                    GeometryArray.TEXTURE_COORDINATE_2,
                                                    2);

        PolygonAttributes polyAttri = new PolygonAttributes();
        polyAttri.setBackFaceNormalFlip(true);
        polyAttri.setCullFace(PolygonAttributes.CULL_FRONT);

        Material materialTex = new Material();
        materialTex.setAmbientColor(0, 0, 0);
        materialTex.setDiffuseColor(1, 1, 1);
        materialTex.setEmissiveColor(0, 0, 0);
        materialTex.setSpecularColor(0, 0, 0);

        TextureAttributes texAttr = new TextureAttributes();
        texAttr.setTextureMode(TextureAttributes.MODULATE);

        // Texture

        faceTextures = new Texture2D[3];        // TODO 3
        faceIconImages = new BufferedImage[3];  // TODO 3
               
        loadTexture(0, "DukeNyaNya512.jpg");
        loadTexture(1, "SwingingDuke512.png");
        loadTexture(2, "bg512.png");

        faceAppear = new Appearance();
        faceAppear.setCapability(Appearance.ALLOW_TEXTURE_WRITE);
        faceAppear.setMaterial(materialTex);        
        faceAppear.setTextureAttributes(texAttr);
        faceAppear.setPolygonAttributes(polyAttri);

        for (Texture2D texture : faceTextures) {
            if (texture != null) {
                faceAppear.setTexture(texture);
                break;
            }
        }

        Shape3D[] facesShape3Ds = new Shape3D[6];
        facesShape3Ds[0] = new Shape3D(cubeFaces[0], faceAppear);
        facesShape3Ds[1] = new Shape3D(cubeFaces[1], faceAppear);
        facesShape3Ds[2] = new Shape3D(cubeFaces[2], faceAppear);
        facesShape3Ds[3] = new Shape3D(cubeFaces[3], faceAppear);
        facesShape3Ds[4] = new Shape3D(cubeFaces[4], faceAppear);
        facesShape3Ds[5] = new Shape3D(cubeFaces[5], faceAppear);

        TransformGroup cubeTG = new TransformGroup();
        Transform3D t3d = new Transform3D();
        t3d.setTranslation(new Vector3d(-1, -1, -1));
        cubeTG.setTransform(t3d);

        Group cubeGroup = new Group();
        for (Shape3D face : facesShape3Ds) {
            cubeGroup.addChild(face);
        }
        cubeTG.addChild(cubeGroup);

        // Viewpoints: 6 views
        createViewpoints(yPositionGroup);
        setupViewpoints(views[0].getFieldOfView());

        PointLight cubeLight = new PointLight();
        cubeLight.setInfluencingBounds(globalBounds);
        cubeLight.setPosition(0.0f, 0.73373f, -0.01396f); // 0, 0.73373, -0.01396
        cubeLight.setAttenuation(0, 0, 0.25f);
        cubeLight.addScope(cubeGroup);

        yPositionGroup.addChild(cubeLight);

        fovBehavior = new FieldOfViewBehavior();
        fovBehavior.setSchedulingBounds(globalBounds);

        sceneBranch.addChild(fovBehavior);
        sceneBranch.addChild(cubeTG);
        sceneBranch.addChild(rotationGroup);
        sceneBranch.addChild(rotInterp);
        sceneBranch.addChild(zPosInterpolator);
        sceneBranch.addChild(yPosInterpolator);
    }

    // Load textures and create images for icons
    private void loadTexture(int index, String fileName) {
         try {
            Texture2D faceTexture = null;
            URL imageUrl = this.getClass().getResource("resources/"+fileName);
            BufferedImage bImage = ImageIO.read(imageUrl);
            if (bImage != null) {
                ImageComponent2D imgComp = new ImageComponent2D(ImageComponent2D.FORMAT_RGB, bImage, false, false);          
                faceTexture = new Texture2D(Texture2D.BASE_LEVEL, Texture2D.RGB, imgComp.getWidth(), imgComp.getHeight());
                faceTexture.setImage(0, imgComp);
                faceTexture.setMagFilter(Texture2D.NICEST);
                faceTexture.setMinFilter(Texture2D.NICEST);
                
                faceTextures[index] = faceTexture;
                
                Image iconImage = bImage.getScaledInstance(32, 32, Image.SCALE_AREA_AVERAGING);
                if (iconImage instanceof BufferedImage) {
                    faceIconImages[index] = (BufferedImage)iconImage;
                }
                // likely
                else {
                    BufferedImage bIcon = new BufferedImage(32, 32, BufferedImage.TYPE_INT_RGB);
                    Graphics2D g2d = bIcon.createGraphics();
                    g2d.drawImage(iconImage, 0, 0, null);
                    faceIconImages[index] = bIcon;
                }
            }        
        }
        catch (IOException e) {
        }
    }

    // Set live
    private void setLive() {
        sceneBranch.compile();
        locale.addBranchGraph(sceneBranch);
        locale.addBranchGraph(viewBranch);
        locale.addBranchGraph(enviBranch);
    }

    // Base world
    private void createUniverse() {
        // Bounds
        globalBounds = new BoundingSphere();
        globalBounds.setRadius(Double.MAX_VALUE);
        //
        // Viewing
        //
        PhysicalBody physicalBody = new PhysicalBody();
        PhysicalEnvironment physicalEnvironment = new PhysicalEnvironment();

        views = new View[6];
        viewTGs = new TransformGroup[6];
        for (int i=0; i < 6; i++) {
            View view = new View();
            view.setPhysicalBody(physicalBody);
            view.setPhysicalEnvironment(physicalEnvironment);
            view.setFieldOfView(Math.toRadians(45));
            views[i] = view;

            viewTGs[i] = new TransformGroup();
            viewTGs[i].setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        }

        //
        // SuperStructure
        //
        VirtualUniverse vu = new VirtualUniverse();
        locale = new Locale(vu);
        //
        // BranchGraphs
        //
        sceneBranch = new BranchGroup();
        viewBranch = new BranchGroup();
        enviBranch = new BranchGroup();

        // EnviBranch

        Background bg = new Background();
        bg.setApplicationBounds(globalBounds);
        bg.setColor(new Color3f(bgColor));

        enviBranch.addChild(bg);
    }

    // fov: radians
    private void setupViewpoints(double fov) {

        Point3d lookAt = new Point3d(0, 0, 0);
        Vector3d upVector = new Vector3d(0.0, 1.0, 0.0);
        Point3d position = new Point3d(0, 0, 0);

        double t = 1.0 / Math.tan(fov / 2.0);

        double dist = t + 1; 

        Transform3D viewT3D = new Transform3D();

        position.set(0, 0, dist); // +Z
        viewT3D.lookAt(position, lookAt, upVector);
        viewT3D.invert();
        viewTGs[0].setTransform(viewT3D);

        position.set(0, 0, -dist); // -Z
        viewT3D.lookAt(position, lookAt, upVector);
        viewT3D.invert();
        viewTGs[1].setTransform(viewT3D);

        position.set(dist, 0, 0); // +X
        viewT3D.lookAt(position, lookAt, upVector);
        viewT3D.invert();
        viewTGs[2].setTransform(viewT3D);

        position.set(-dist, 0, 0); // -X
        viewT3D.lookAt(position, lookAt, upVector);
        viewT3D.invert();
        viewTGs[3].setTransform(viewT3D);

        // Top
        upVector.set(0.0, 0.0, -1.0);

        position.set(0, dist, 0); // +Y
        viewT3D.lookAt(position, lookAt, upVector);
        viewT3D.invert();
        viewTGs[4].setTransform(viewT3D);

        // Bottom
        upVector.set(0.0, 0.0, 1.0);

        position.set(0, -dist, 0); // -Y
        viewT3D.lookAt(position, lookAt, upVector);
        viewT3D.invert();
        viewTGs[5].setTransform(viewT3D);
    }

    private void createViewpoints(TransformGroup tuxScope) {

        for (int i=0; i < 6; i++) {
            ViewPlatform vp = new ViewPlatform();
            views[i].attachViewPlatform(vp);

            DirectionalLight headLight = new DirectionalLight();
            headLight.setInfluencingBounds(globalBounds);
            headLight.addScope(tuxScope);

            ViewSpecificGroup vsg = new ViewSpecificGroup();
            vsg.addView(views[i]);
            vsg.addChild(headLight);

            viewTGs[i].addChild(vp);
            viewTGs[i].addChild(vsg);

            viewBranch.addChild(viewTGs[i]);
        }
    }

    // Array of FXCanvas3DMVs and FXCanvas3DMVControl
    private void createFXCanvas3DMV() {
        // Arbitrary size
        Dimension dim = new Dimension(200, 200);
        GraphicsConfigTemplate3D gCT = new GraphicsConfigTemplate3D();

        // At first: array of FXCanvas3DMV and FXCanvas3DMVControl
        // FXCanvas3DMV needs a FXCanvas3DMVControl to work properly!
        fxCanvas3DMVs = new FXCanvas3DMVSquare[6];
        for (int i=0; i < 6; i++) {
            FXCanvas3DMVSquare fxCanvas3DMV = new FXCanvas3DMVSquare(gCT);
            fxCanvas3DMV.setName("FXCanvas3DMV_" + i);
            fxCanvas3DMVs[i] = fxCanvas3DMV;
        }
        fxC3DMVControl = new FXCanvas3DMVControl(fxCanvas3DMVs);

        // Secondly: size, parent, off-screen Canvas3D
        for (int i = 0; i < 6; i++) {
            FXCanvas3DMV fxCanvas3DMV = fxCanvas3DMVs[i];

            fxCanvas3DMV.setPreferredSize(dim);
            fxCanvas3DMV.setSize(dim);

            // Provide a parent
            JPanel jPanel = new JPanel(new BorderLayout());
            jPanel.add(fxCanvas3DMV, BorderLayout.CENTER);

            // Off-screen Canvas3D of this lightweight JCanvas3D
            Canvas3D offCanvas3D = fxCanvas3DMV.getOffscreenCanvas3D();
            if (offCanvas3D != null) {
                // View renders into the off-screen Canvas3D
                views[i].addCanvas3D(offCanvas3D);
            }
            else {
                System.out.println("TuxInTheBoxUniverse: OffscreenCanvas3D = null !!");
                System.exit(0);
            }
        }
    }

    // Set viewpoint and fov in behavior scheduler
    // Sync of both requires two posts !!
    private final class FieldOfViewBehavior extends Behavior {

        static final int UPDATE = 1;
        static final int RECALL = 2;
        private WakeupOnBehaviorPost updatePost = new WakeupOnBehaviorPost(this, UPDATE);
        private WakeupOnBehaviorPost recallPost = new WakeupOnBehaviorPost(this, RECALL);
        private double fov = Math.toRadians(45.0);

        FieldOfViewBehavior() {
        }

        @Override
        public void initialize() {
            wakeupOn(updatePost);
        }

        @Override
        public void processStimulus(Enumeration criteria) {

            // TODO
            if (views == null || views[0] == null || viewTGs == null || viewTGs[0] == null) {
                return;
            }

            WakeupOnBehaviorPost post = null;

            while (criteria.hasMoreElements()) {

                post = (WakeupOnBehaviorPost)criteria.nextElement();

                if (post.getPostId() == UPDATE) {
                    setupViewpoints(fov);

                    this.postId(RECALL);
                    wakeupOn(recallPost);
                }
                // RECALL
                else {
                    for (int i=0; i < 6; i++) {
                        views[i].setFieldOfView(fov);
                    }
                    isFovActive = false;
                    wakeupOn(updatePost);
                }
            }
        }
    }

    private final class FXCanvas3DMVSquare extends FXCanvas3DMV {

        private FXCanvas3DMVSquare(GraphicsConfigTemplate3D gCT) {
            super(gCT);
        }

        // Set bounds only if it is of square size
        @Override
        public void setBounds(int x, int y, int width, int height) {
            if (width == height) {
                super.setBounds(x, y, width, height);
            }
        }
    }
}
