package org.interactivemesh.scala.j3d.samples.propeller

// Java
import java.awt.{Color, GraphicsDevice} 
import java.io.{FileNotFoundException, IOException}

import java.util.{Enumeration => JEnumeration}

// Java 3D
import javax.media.j3d.{Alpha, Background, Behavior, BoundingSphere, BranchGroup,
  Canvas3D, DirectionalLight, GraphicsConfigTemplate3D, ImageComponent, ImageComponent2D,
  Locale, PhysicalBody, PhysicalEnvironment, Shape3D, Transform3D, TransformGroup,
  View, ViewPlatform, ViewSpecificGroup, VirtualUniverse, WakeupCondition,
  WakeupOnBehaviorPost, WakeupOnElapsedFrames, WakeupOr}

import javax.vecmath.{Color3f, Point3d, Vector3d}

// X3DModelImporter
import com.interactivemesh.j3d.interchange.ext3d.XImportException
import com.interactivemesh.j3d.interchange.ext3d.XShapeReader

// OrbitBehaviorInterim 2.1, see http://www.interactivemesh.org/testspace/orbitbehavior.html
import com.interactivemesh.j3d.community.utils.navigation.orbit.OrbitBehaviorInterim

/*
 * PropellerUniverse.scala
 *
 * Version: 1.2
 * Date: 2011/05/26
 *
 * Copyright (c) 2010
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'PropellerUniverse.scala'.
 *
 * See also license notice above: j3d-examples: src.resources.images.bg.jpg
 */
		
private final class PropellerUniverse(private val panel: PropellerPanel, gd: GraphicsDevice) {
  
  private val bgColor0 = new Color(0, 102, 204)
  private val globalBounds = new BoundingSphere(new Point3d, java.lang.Double.MAX_VALUE)
  
  // Viewpoints
  private[propeller] object ViewPoint extends Enumeration {
    type ViewPoint = Value
    val Front = Value("Front")
    val OnRod = Value("OnRod")
    val Rod = Value("Pistonrod")
    val Piston = Value("Piston")
  }
  
  // Vantage points
  private abstract class VantagePoint(private val name: String) {
    val position = new Point3d
    val lookAt = new Point3d
    val upVector = new Vector3d(0.0, 1.0, 0.0)
    val rotCenter = new Point3d  
//    var fov = swing.math.PI/4
    
    override def toString = name
    
    def applyTo(navigator: OrbitBehaviorInterim) =
      navigator.setViewingTransform(position, lookAt, upVector, rotCenter)
  }
  
  import ViewPoint._
  
  // Map : ViewPoint -> VantagePoint
  private val vantagePoints = Map(
    Front -> new VantagePoint(Front.toString) {
      position.set(0.72, 0, 1.73)
      lookAt.set(-0.25, 0, -0.0525)
      rotCenter.set(0, 0, -0.0525)
    },
    OnRod -> new VantagePoint(OnRod.toString) {
      position.set(0.09, 0, -0.46)
      lookAt.set(0.09, 0, 0)
      rotCenter.set(0, 0, -0.0525)
    },
    Rod -> new VantagePoint(Rod.toString) {
      position.set(-0.159, 0, -0.268)
      lookAt.set(0.045, 0, -0.1275)
      rotCenter.set(0.045, 0, -0.1275)
    },
    Piston -> new VantagePoint(Piston.toString) {
      position.set(0.1087, 0.023, -0.1038)
      lookAt.set(0, 0, -0.10378)
      rotCenter.set(0, 0, -0.10378)
    }
  )
 
  //
  // SuperStructure
  //
  
  private val vu = new VirtualUniverse
  private val locale = new Locale(vu)

  //
  // Viewing
  //

  private val phBody = new PhysicalBody
  private val phEnvironment = new PhysicalEnvironment
  
  private final class Viewer extends View {
    setPhysicalBody(phBody)
    setPhysicalEnvironment(phEnvironment)
  }
  
  // Map : ViewPoint -> View
  private val views = Map(
    Front -> new Viewer,
    OnRod -> new Viewer,
    Rod -> new Viewer,
    Piston -> new Viewer
  )
  
  // GraphicsDevise from JFrame/JApplet
  private val gcfg = gd.getBestConfiguration(new GraphicsConfigTemplate3D)

  // Map : ViewPoint -> Canvas3D
  private val canvas3Ds = Map(
    Front -> new Canvas3D(gcfg),
    OnRod -> new Canvas3D(gcfg),
    Rod -> new Canvas3D(gcfg),
    Piston -> new Canvas3D(gcfg)
  )
  
  import OrbitBehaviorInterim._
  
  private final class Navigator(vp: ViewPoint) extends OrbitBehaviorInterim(REVERSE_ALL) {
    setSchedulingBounds(globalBounds)
    setClippingEnabled(true)
    // canvas3D, view, viewTG
    setAWTComponent(canvas3Ds.get(vp).get)
    setVpView(views.get(vp).get)
    // ViewTG see below
  }
  // Map : ViewPoint -> Navigator
  private val navigators = Map(
    Front -> new Navigator(Front),
    OnRod -> new Navigator(OnRod),
    Rod -> new Navigator(Rod),
    Piston -> new Navigator(Piston)
  )
  
  //
  // BranchGraphs
  //
  
  private val enviBranch = new BranchGroup
  private val sceneBranch = new BranchGroup
  private val viewBranch = new BranchGroup
  
  //
  // View branch
  //

  // ViewPoint specific TransformGroup subscene
  private final class ViewTG(vp: ViewPoint) extends TransformGroup {
    setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE) 
    private val p = new ViewPlatform
    private val vsg = new ViewSpecificGroup
	
    private val view = views.get(vp).get
    view.attachViewPlatform(p)
    view.addCanvas3D(canvas3Ds.get(vp)get)
	
    navigators.get(vp).get.setViewingTransformGroup(this)
	
    addChild(p)
    addChild(navigators.get(vp).get)
    addChild(new ViewSpecificGroup {
      addView(view)
      addChild( new DirectionalLight { setInfluencingBounds(globalBounds)} )
    })
  }
  
  viewBranch.addChild(new ViewTG(Front))
  viewBranch.addChild(new ViewTG(Rod))
  viewBranch.addChild(new ViewTG(Piston))

  //
  // Scene branch
  //
  
    // See 'propellerBehavior'
  val crankarmTG = new TransformGroup {
    setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE)
  }
  // See 'propellerBehavior'  
  val cylinderTG = new TransformGroup {
    setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE)
  }  
  // See 'propellerBehavior'
  val propellerTG = new TransformGroup {
    setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE)
  }
    
  try {
	// X3D importer
    val x3dImporter = new XShapeReader();
    // Read 
    x3dImporter.read(this.getClass().getResource("resources/PropellerModelOCC-AllShapes.x3d"));
  
    // Typesafe retrieving of shapes (no check if null)
    val x3dShapes = x3dImporter.getImport();
    // Map : shape's name -> shape
    var namedShapes = Map[String, Shape3D]()
    for (shape <- x3dShapes) {
	  namedShapes += (shape.getName() -> shape) 
    }
  
    x3dImporter.close();

    crankarmTG.addChild(namedShapes("Crankarm"))
    crankarmTG.addChild(new ViewTG(OnRod))

    cylinderTG.addChild(crankarmTG)
    cylinderTG. addChild(namedShapes("PistonWristPin"))
    cylinderTG.addChild(namedShapes("Piston"))
    
    propellerTG.addChild(namedShapes("PropAxis"))
    propellerTG.addChild(new TransformGroup {
      setTransform(new Transform3D { rotZ(Math.toRadians(45)) })
      addChild(namedShapes("Propeller"))
    })
    	
    // Rotation Behavior
    sceneBranch.addChild(propellerBehavior) 
    // Propeller assembly
    sceneBranch.addChild(cylinderTG)
    sceneBranch.addChild(propellerTG)
    sceneBranch.addChild(namedShapes("EngineBlock"))
    sceneBranch.addChild(namedShapes("CylinderHead"))
  }
  catch {
    case e: XImportException => e.printStackTrace
  }
  
  //
  // Environment branch
  //

  private val background = new Background {
    setCapability(Background.ALLOW_IMAGE_WRITE)
    setApplicationBounds(globalBounds)
    setColor(new Color3f(bgColor0))
  }

  private val imageUrl = this.getClass.getResource("resources/bg.jpg")
  try {
    val bgImage = javax.imageio.ImageIO.read(imageUrl)
    if (bgImage ne null) {
      background.setImageScaleMode(Background.SCALE_FIT_ALL)
      background.setImage(new ImageComponent2D(ImageComponent.FORMAT_RGB, bgImage, false, false))
    }
  }
  catch {
    case e: IOException => {}
  }
        
  enviBranch.addChild(background)
  
  //
  // Set live
  //
  locale.addBranchGraph(sceneBranch)
  locale.addBranchGraph(viewBranch)
  locale.addBranchGraph(enviBranch)

  //
  // Setup navigation
  //
  
  private var sceneRadius = 1.0d

  private var sphereBounds = new BoundingSphere
  private val bounds = sceneBranch.getBounds
  if (!bounds.isEmpty) {
    if (bounds.isInstanceOf[BoundingSphere])
      sphereBounds = bounds.asInstanceOf[BoundingSphere]
    else
      sphereBounds = new BoundingSphere(bounds)
  }

  sceneRadius = sphereBounds.getRadius
	
  navigators foreach (vk => {
    vk._2.setTransFactors(sceneRadius/4.0f, sceneRadius/4.0f)
    vk._2.setZoomFactor(sceneRadius/4.0f)
    vk._2.setRotFactors(0.4f, 0.4f)
    vk._2.setClippingBounds(sphereBounds)
  })
  
  // Set all viewpoints
//  for (vp <- ViewPoint) vantagePoint(vp)  TODO: Bug since 2.9.0 !?  
  ViewPoint.values foreach {vp => vantagePoint(vp)}
  
  //
  // Universe interaction
  //
  
  private[propeller] def canvas3D(vp: ViewPoint): Canvas3D = canvas3Ds.get(vp).get

  private[propeller] def enableNavigator(vp: ViewPoint, b: Boolean): Unit = 
	navigators.get(vp).get.setEnable(b)
  
  private[propeller] def vantagePoint(vp: ViewPoint): Unit =
    vantagePoints.get(vp).get.applyTo(navigators.get(vp).get)
    
  private[propeller] def rotationSpeed(speed: Int): Unit = propellerBehavior.rotationSpeed(speed)

  private[propeller] def closeUniverse: Unit = {        
    views foreach (vk => {
      vk._2.removeAllCanvas3Ds
      vk._2.attachViewPlatform(null)
    })
    vu.removeAllLocales
  }
        
  //
  // Rotation: direction, speed
  //
  private object propellerBehavior extends Behavior {

    setSchedulingBounds(globalBounds)
	  
    // FPS
    private var lastTimeFPS: Long       = System.nanoTime
    private var frameCounter: Int       = 0
    private var elapsedFrames: Int      = 10
    private var isUpdateFPS             = false

    // Model
    private val onePi: Float            = Math.Pi.asInstanceOf[Float]
    private val twoPi: Float            = Math.Pi.asInstanceOf[Float] * 2

    private var axisRadians: Float      = 0f
    private var sinAxisRadians: Float   = 0f
    private var crankarmRadians: Float	= 0f

    private val radius: Float           = 0.045f
    private val crankarmLength: Float	= 0.120f
    private val dist: Float             = 0.165f // 0.045 + 0.0120
    private var currDist: Float         = 0.165f
    private var transX: Float           = 0f
	
    // Transformation
    private val cylinderTransform       = new Transform3D
    private val cylinderTranslation     = new Vector3d

    private val propellerRotation       = new Transform3D

    private val crankarmTransform       = new Transform3D
    private val crankarmRotation        = new Transform3D
    private val crankarmCenterTrans     = new Transform3D {
      setTranslation(new Vector3d(-0.165, 0, 0))
    }
    private val crankarmCenterTransInv  =   new Transform3D {
      setTranslation(new Vector3d(0.165, 0, 0))
    }
    private val crankarmTemp            =   new Transform3D
    
    // "Alpha"
    private val NsecPerMsec: Long     = 1000000 // nano sec per micro sec
    private val MinDuration: Long     = 100L // 0.1 sec per rotation
    private var startTime: Long       = 0L
    private var duration: Long        = 0L
    private var currAlphaValue: Float = 0f
    private var loops: Float          = 0f
            
    // WakeupCriterions
    private var speedPost = new WakeupOnBehaviorPost(this, 0)
    private val activeFramesOrPost = new WakeupOr(
      Array(new WakeupOnBehaviorPost(this, 0), new WakeupOnElapsedFrames(0))
    )
    
    private def enableFPS(enable: Boolean) {        	
      isUpdateFPS = enable
      frameCounter = 0
      elapsedFrames = 10
      lastTimeFPS = System.nanoTime
    }
    
    // Speed, s = [0, 100]	  
    private[PropellerUniverse] def rotationSpeed(s: Int) = postId(s)
    
    //
    // Behavior
    //
    
    override def initialize = wakeupOn(speedPost)
	
    override def processStimulus(criteria: JEnumeration[_]) { // renamed java.util.Enumeration
      while (criteria.hasMoreElements) {    	  
        criteria.nextElement match {
        	
          case w: WakeupOnBehaviorPost =>   
          
          	val speed = w.getTriggeringPostId // [0, 100]
          	
            // Stop rotation
            if (speed == 0) { 
              enableFPS(false)
              panel.updateFPS(0)
              panel.updateRPM(0)
              wakeupOn(speedPost)   
              return
            }
            // Set speed: slider [1, 100] -> [10, 0.1] sec per rotation
            else {
              if (!isUpdateFPS)
                enableFPS(true)
               
              // duration -> speed
    	      duration = (NsecPerMsec * MinDuration * 100f/speed).asInstanceOf[Long]   	  
              // startTime -> sync
    	      startTime = System.nanoTime - (currAlphaValue * duration).asInstanceOf[Long]

              // 60 X 1000ms / duration in sec
              panel.updateRPM( (60000d / (duration/NsecPerMsec)).asInstanceOf[Int] )
              
              wakeupOn(activeFramesOrPost)
            }

          case w: WakeupOnElapsedFrames =>		
            // FPS
            frameCounter += 1        
            if (frameCounter >= elapsedFrames) {
            
              val currTimeFPS = System.nanoTime
              val frameDuration = (currTimeFPS - lastTimeFPS)/frameCounter
            
              val fps = NsecPerMsec * 1000d / frameDuration
            
              lastTimeFPS = currTimeFPS

              frameCounter = 0
              elapsedFrames = Math.max(1, ((fps + 0.5) / 4)).asInstanceOf[Int] // 4 times per second
            
              panel.updateFPS(fps.asInstanceOf[Int])
            }

            // Rotation  
            
            // [0, 1] : fraction of current loop
            //currAlphaValue = ((System.nanoTime - startTime) % duration) / duration  
            loops = (System.nanoTime - startTime) 
            loops /= duration
            currAlphaValue = loops - loops.asInstanceOf[Int]
            
            // [0, twoPi]
            axisRadians = twoPi * currAlphaValue 

            sinAxisRadians = Math.sin(axisRadians).asInstanceOf[Float]

            // Law of sines

            crankarmRadians = Math.asin(radius * sinAxisRadians / crankarmLength).asInstanceOf[Float]

            // If currRadians != onePi | twoPi | 0
            if (sinAxisRadians != 0.0f) {
              currDist = (crankarmLength / sinAxisRadians * Math.sin(onePi-axisRadians-crankarmRadians)).asInstanceOf[Float]
            }
            else {
              if (axisRadians == onePi) 
                currDist = crankarmLength - radius
              else 
                currDist = crankarmLength + radius
            }

            transX = dist - currDist
        
            cylinderTranslation.setX(-transX)
            cylinderTransform.setTranslation(cylinderTranslation)
            cylinderTG.setTransform(cylinderTransform)

            // Rotates in (0, 0, 0)
            propellerRotation.rotZ(axisRadians)
            propellerTG.setTransform(propellerRotation)

            // Rotates in (165, 0, 0)
            crankarmRotation.rotZ(-crankarmRadians)
            crankarmTemp.mul(crankarmRotation, crankarmCenterTrans)
            crankarmTransform.mul(crankarmCenterTransInv, crankarmTemp)
            crankarmTG.setTransform(crankarmTransform)
              
            wakeupOn(activeFramesOrPost)
            
        } // End of match
      } // End of while       
    } 
  }
  
  // Java 3D properties
  private[propeller] def printJava3DProps {

    val vuProps = VirtualUniverse.getProperties

    println("Java 3D Version  =  " + vuProps.get("j3d.version"))
    println("Java 3D Renderer  =  " + vuProps.get("j3d.renderer"))
    println("Java 3D Pipeline  =  " + vuProps.get("j3d.pipeline"))
    println("------------------------------------------------------------------------")
  
    val c3dProps = canvas3Ds.get(Front).get.queryProperties

    println("Native Version  =  " + c3dProps.get("native.version"))
    println("Native GLSL  =  " + c3dProps.get("shadingLanguageGLSL"))
    println("Native Vendor  =  " + c3dProps.get("native.vendor"))
    println("Native Renderer  =  " + c3dProps.get("native.renderer"))

    println("------------------------------------------------------------------------")
    println("")
  }
  
}
