package com.interactivemesh.j3d.testspace.jcanvas3d;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Toolkit;

import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import java.util.ArrayList;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JLayeredPane;
import javax.swing.JPanel;
import javax.swing.JSlider;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * Stereoscopic3DPanel.java
 *
 * Version: 0.5
 * Date: 2009/04/29
 *
 * Copyright (c) 2009
 * August Lammersdorf, InteractiveMesh e.K.
 * Kolomanstrasse 2a, 85737 Ismaning
 * Germany / Munich Area
 * www.InteractiveMesh.com/org
 *
 * Please create your own implementation.
 * This source code is provided "AS IS", without warranty of any kind.
 * You are allowed to copy and use all lines you like of this source code
 * without any copyright notice,
 * but you may not modify, compile, or distribute this 'Stereoscopic3DPanel.java'.
 *
 */
final class Stereoscopic3DPanel {  

    private Font titleFont = null; 
    private Font textFont = null; 
    
    private Color bgColor = new Color(0.0f, 0.4f, 0.8f);        // blue
    private Color fgColor = new Color(1.0f, 1.0f, 1.0f);        // white
    
    private Color enterColor = new Color(1.0f, 0.7f, 0.0f);     // orange
    private Color pressColor = new Color(1.0f, 0.2f, 0.0f);     // dark orange            
    private Color selectColor = new Color(0.0f, 1.0f, 0.6f);    // neon green
    
    private int borderBLR = 50;
    private int borderT   = 80;
        
    private int space = 10;
    private int vspace = 5;
    
    private Slider rotationSlider = null;
    private Slider diatanceSlider = null;
    
    private Stereoscopic3D universe = null;
    
    
    Stereoscopic3DPanel() {
        
    }
    
    // JApplet
    void destroy() {
        universe.closeUniverse();
    }
    
    void addTo(Container container, final JLayeredPane jLayeredPane) {
        
        System.out.println("Stereoscopic 3D is running.");
        
        Dimension screenDim = Toolkit.getDefaultToolkit().getScreenSize();
        
        int screenHeight = screenDim.height;
        
        // screenHeight >= 1200
        int textFontSize = 20;
        int titleFontSize = 30;
        borderBLR = 50;
        borderT = 80;
        space = 10;

        // screenHeight  < 1024
        if (screenHeight < 1024) {
            textFontSize = 16;
            titleFontSize = 26;
            borderBLR = 30;
            borderT = 50;
            space = 5;
        }
        // 1024 <= screenHeight < 1200
        else if (screenHeight < 1200) {
            textFontSize = 18;
            titleFontSize = 28;
            borderBLR = 40;
            borderT = 60;
            space = 8;
        }

        titleFont = new Font("Dialog", Font.PLAIN, titleFontSize); 
        textFont = new Font("Dialog", Font.BOLD, textFontSize); 

            
        universe = new Stereoscopic3D();
        
       
        // Parent of JCanvas3DAnaglyph
        JPanel jCanvas3D = universe.getJCanvas3D();        
        
        //
        // Title
        //
        JLabel titleLabel = new JLabel();
        titleLabel.setFont(titleFont);
        titleLabel.setForeground(fgColor);
        titleLabel.setText("JCanvas3D Anaglyph");
        titleLabel.setMinimumSize(new Dimension(100, borderT));
        titleLabel.setPreferredSize(new Dimension(100, borderT));
        titleLabel.setMaximumSize(new Dimension(Integer.MAX_VALUE, borderT));
        
        JPanel titlePanel = new JPanel();
        titlePanel.setLayout(new BoxLayout(titlePanel, BoxLayout.X_AXIS));
        titlePanel.setBackground(bgColor);
        titlePanel.setOpaque(false);
        titlePanel.setBorder(BorderFactory.createEmptyBorder(0, 2*space, 0, 0));
        
        titlePanel.add(titleLabel);
        titlePanel.add(Box.createHorizontalGlue());
                
        //
        // Controls
        //
        
        // VantagePoint
        
        JPanel vpPanel = new YPanel() {
            @Override
            public void paintComponent(Graphics g) {               
                super.paintComponent(g);                
                Dimension size = this.getSize();
                g.setColor(Color.WHITE);                
                // Horizontal line
                g.drawLine(0, size.height/2, size.width, size.height/2);
            }
        };        

        ButtonLabel frontButtom = new ButtonLabel("Front", 0, 5, 0, 5, "Front viewpoint");       
        ButtonLabel topButton = new ButtonLabel("Top", 0, 5, 0, 5, "Top viewpoint");
        
        ButtonLabelGroup vpGoup = new ButtonLabelGroup() {
            @Override
            void action(ButtonLabel button) {
                universe.setVantagePoint(button.getText());
            }           
        };
        vpGoup.add(frontButtom);
        vpGoup.add(topButton);
        
        vpPanel.add(frontButtom);
        vpPanel.add(Box.createVerticalStrut(vspace));
        vpPanel.add(topButton);
       
        // Cube dim
        
        JPanel cubePanel = new YPanel() {
            @Override
            public void paintComponent(Graphics g) {
                super.paintComponent(g);
                
                Dimension size = this.getSize();
                g.setColor(Color.WHITE);                
                // Horizontal line
                g.drawLine(0, size.height/2, size.width, size.height/2);
            }
        };
        
        String tip = " Tux per row / column";
        
        RadioLabel[] cubeRadios = new RadioLabel[2];
        cubeRadios[0] = new RadioLabel(String.valueOf(2), 0, 5, 0, 5, String.valueOf(2)+tip);
        cubeRadios[1] = new RadioLabel(String.valueOf(3), 0, 5, 0, 5, String.valueOf(3)+tip);

        RadioLabelGroup cubeGroup = new RadioLabelGroup() {
            @Override
            void selectionChanged(RadioLabel radio) {
                universe.setCubeDimension(Integer.parseInt(radio.getText())); 
            }
        };
        
        cubeGroup.add(cubeRadios[0]);
        cubeGroup.add(cubeRadios[1]);
        
        cubeGroup.setSelection(cubeRadios[0]);
        
        cubePanel.add(cubeRadios[0]);
        cubePanel.add(Box.createVerticalStrut(vspace));
        cubePanel.add(cubeRadios[1]);
        
        
        // Rotation
               
        JPanel rotationPanel = new YPanel();

        SingleButtonLabel rotationButtonLabel = new SingleButtonLabel(" <   Rotation   > ", "Reset rotation") {
            @Override
            void action() {
                universe.resetRotation();
                rotationSlider.setAtValue(50);
            }
        };
        
        Dimension rotLabelSize = rotationButtonLabel.getPreferredSize();
        Dimension sliderSize = new Dimension((int)(rotLabelSize.width*1.2), rotLabelSize.height);
        
        rotationSlider = new Slider(0, 100, 50, sliderSize) { 
            @Override
            void valueChanged(int value) {  
                universe.setRotationSpeed(value);            
            }
        };        
        
        rotationPanel.add(rotationButtonLabel);
        rotationPanel.add(Box.createVerticalGlue());
        rotationPanel.add(rotationSlider);
               
        // Interpupillary distance
        
        JPanel diatancePanel = new YPanel();

        final int defaultDist = 66; // mm
        
        SingleButtonLabel diatanceButtonLabel = new SingleButtonLabel("Distance", "Interpupillary Distance") {
            @Override
            void action() {
                diatanceSlider.setValue(defaultDist);
            }
        };

        diatanceSlider = new Slider(0, 120, defaultDist, sliderSize) { 
            @Override
            void valueChanged(int value) {  
                universe.setEyesPosition(value);            
            }
        };        
       
        diatancePanel.add(diatanceButtonLabel);
        diatancePanel.add(Box.createVerticalGlue());
        diatancePanel.add(diatanceSlider);
        
        // Knot transparency
        
        JPanel transPanel = new YPanel();

        Label transButtonLabel = new Label("Transparency", "Knot transparency");

        Slider transSlider = new Slider(0, 100, 0, sliderSize) { 
            @Override
            void valueChanged(int value) {  
                universe.setKnotTransparency(value);            
            }
        };        
        
        transPanel.add(transButtonLabel);
        transPanel.add(Box.createVerticalGlue());
        transPanel.add(transSlider);
        
        //
        // Action panel
        //
        final JPanel actionPanel = new JPanel();
        actionPanel.setLayout(new BoxLayout(actionPanel, BoxLayout.X_AXIS));
        actionPanel.setOpaque(false);
        
        actionPanel.add(vpPanel);
        actionPanel.add(Box.createHorizontalStrut(2*space));
        actionPanel.add(cubePanel);
        actionPanel.add(Box.createHorizontalStrut(2*space));
        actionPanel.add(rotationPanel);
        actionPanel.add(Box.createHorizontalStrut(2*space));
        actionPanel.add(transPanel);
        actionPanel.add(Box.createHorizontalStrut(2*space));
        actionPanel.add(diatancePanel);
        
        Dimension actionSize = actionPanel.getPreferredSize();
        final int actionWidth = actionSize.width;
        final int actionHeight = actionSize.height;
                
        // Initial size
        int panelDim = (int)(Math.min(screenDim.width, screenDim.height) * 0.7f);       
        jCanvas3D.setPreferredSize(new Dimension((int)(panelDim*1.5f), panelDim));
        
        // Frame
        
        final JPanel framePanel = new JPanel();
        framePanel.setLayout(new BoxLayout(framePanel, BoxLayout.Y_AXIS));
        framePanel.setBackground(bgColor);
        framePanel.setOpaque(false);
        framePanel.setBorder(BorderFactory.createEmptyBorder(0, borderBLR, borderBLR, borderBLR));
        
        framePanel.add(titlePanel);
        
        // ActionPanel in PALETTE_LAYER
        
        jCanvas3D.addComponentListener(new ComponentAdapter() { 
            public void componentResized(ComponentEvent event) {
                Dimension size = jLayeredPane.getSize();
                actionPanel.setBounds((size.width - actionWidth)/2, size.height-actionHeight-2*space, 
                                      actionWidth, actionHeight);     
                framePanel.setBounds(0, 0, size.width, size.height);     
                framePanel.revalidate(); 
                framePanel.repaint();
            }
        });
        
        jLayeredPane.add(framePanel, JLayeredPane.PALETTE_LAYER);
        jLayeredPane.add(actionPanel, JLayeredPane.PALETTE_LAYER);
               
        container.add(jCanvas3D);

    }
    
    //
    // GUI components
    //

    final private class XPanel extends JPanel {
        private XPanel() {
            setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
            setOpaque(false);
            setAlignmentY(JPanel.CENTER_ALIGNMENT);
        }
    }
    
    private class YPanel extends JPanel {
        private YPanel() {
            setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
            setOpaque(false);
            setAlignmentX(JPanel.CENTER_ALIGNMENT);
        }
    }
    
    private class Label extends JLabel {
        
        private boolean isPressed = false;
        private boolean isSelected = false;            

        
        private Label(String text) {
            this(text, null);
        }
        private Label(String text, String tip) {
            super(text);
            setFont(textFont);
            setForeground(fgColor);
            setAlignmentX(JPanel.CENTER_ALIGNMENT);
            if (tip != null)
                this.setToolTipText(tip);
        }
        
        boolean isPressed() {
            return isPressed; 
        }
        void setPressed(boolean press) {
            isPressed = press;
        }
        
        boolean isSelected() {
            return isSelected; 
        }       
        void setSelected(boolean select) {
            isSelected = select;
        }
    }
    
    // Slider
    
    private class Slider extends JSlider {       
        
        private ChangeListener listener = null;
        
        private Slider(int min, int max, int value, Dimension preferredSize) {
            super(min, max, value);
            setOpaque(false);
            setAlignmentX(JPanel.CENTER_ALIGNMENT);
            setBorder(BorderFactory.createEmptyBorder());
            setPreferredSize(preferredSize);
            
            listener = new ChangeListener() {
                public void stateChanged(ChangeEvent event) {
                    valueChanged(Slider.this.getValue());
                }
            };            
            addChangeListener(listener);
        }
        
        private void setAtValue(int value) {
            this.removeChangeListener(listener);
            this.setValue(value);
            this.addChangeListener(listener);
        }
        
        void valueChanged(int value) {           
        }
    }
    
    // Button / RadioButton
    
    //
    // Button / -Group
    //
    // Used in a ButtonGroup
    private class ButtonLabel extends Label {
    
        private ButtonLabel(String text) {
            this(text, 0, 0, 0, 0, null);
        }
        private ButtonLabel(String text, String tip) {
            this(text, 0, 0, 0, 0, tip);
        }
        private ButtonLabel(String text, int t, int l, int b, int r, String tip) {
            super(text);
            setBorder(BorderFactory.createEmptyBorder(t, l, b, r));           
            if (tip != null)
                this.setToolTipText(tip);
        }
    }
    // Individual button
    private class SingleButtonLabel extends ButtonLabel {
        
        private SingleButtonLabel(String text) {
            this(text, 0, 0, 0, 0, null);
        }
        private SingleButtonLabel(String text, String tip) {
            this(text, 0, 0, 0, 0, tip);
        }
        private SingleButtonLabel(String text, int t, int l, int b, int r, String tip) {
            super(text, t, l, b, r, tip);
            
            addMouseListener(new MouseAdapter() { 

                public void mouseEntered(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    if (!button.isPressed())
                        button.setForeground(enterColor);
                }
                public void mouseExited(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    if (!button.isPressed())
                        button.setForeground(fgColor);
                }
                public void mousePressed(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    button.setPressed(true);
                    button.setForeground(pressColor);
                }
                public void mouseReleased(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    button.setPressed(false);
                    if (button.contains(event.getX(), event.getY())) {
                        button.setForeground(enterColor);                                  
                        action();
                    }
                    else
                        button.setForeground(fgColor);
                }
            });
        }
    
        void action() {           
        }
    }
    
    private class ButtonLabelGroup {
        
        // ButtonLabels of this group 
        private ArrayList<ButtonLabel> buttonList = null;
        
        // Single listener for all ButtonLabels
        private MouseAdapter actionListener = null;
        
        private ButtonLabelGroup() {
                       
            buttonList = new ArrayList<ButtonLabel>();
            
            actionListener = new MouseAdapter() { 

                public void mouseEntered(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    if (!button.isPressed())
                        button.setForeground(enterColor);
                }
                public void mouseExited(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    if (!button.isPressed())
                        button.setForeground(fgColor);
                }
                public void mousePressed(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    button.setPressed(true);
                    button.setForeground(pressColor);
                }
                public void mouseReleased(MouseEvent event) {
                    ButtonLabel button = (ButtonLabel)event.getSource();
                    button.setPressed(false);
                    if (button.contains(event.getX(), event.getY())) {
                        button.setForeground(enterColor);                                  
                        action(button);
                    }
                    else
                        button.setForeground(fgColor);
                }
            };
        }
        
        // Use 'SingleButtonLabel' if an individual action is required as well
        private void add(ButtonLabel button) {
            buttonList.add(button);
            button.addMouseListener(actionListener);
        }
        
        void action(ButtonLabel button) {
        }
    }
    
    //
    // RadioButton / -Group
    //   
    final private class RadioLabel extends Label {
        
        private RadioLabel(String text) {
            this(text, 0, 0, 0, 0, null);
        }
        private RadioLabel(String text, int t, int l, int b, int r, String tip) {
            super(text);
            setBorder(BorderFactory.createEmptyBorder(t, l, b, r));    
            if (tip != null)
                this.setToolTipText(tip);
        }
    }
    
    private class RadioLabelGroup {
        
        // RadioLabels of this group 
        private ArrayList<RadioLabel> radioList = null;
        
        // Single listener for all RadioLabels
        private MouseAdapter actionListener = null;
        
        private RadioLabelGroup() {
                       
            radioList = new ArrayList<RadioLabel>();
            
            actionListener = new MouseAdapter() { 
                               
                @Override
                public void mouseEntered(MouseEvent event) {
                    RadioLabel radio = (RadioLabel)event.getSource();
                    if (!radio.isPressed() && !radio.isSelected())
                        radio.setForeground(enterColor);
                }
                @Override
                public void mouseExited(MouseEvent event) {
                    RadioLabel radio = (RadioLabel)event.getSource();
                    if (!radio.isPressed() && !radio.isSelected())
                        radio.setForeground(fgColor);
                }
                @Override
                public void mousePressed(MouseEvent event) {
                    RadioLabel radio = (RadioLabel)event.getSource();
                    if (!radio.isSelected()) {
                        radio.setPressed(true);
                        radio.setForeground(pressColor);
                    }
                }
                @Override
                public void mouseReleased(MouseEvent event) {
                    RadioLabel radio = (RadioLabel)event.getSource();
                    
                    if (radio.isSelected())
                        return;
                    
                    radio.setPressed(false);
                    if (radio.contains(event.getX(), event.getY())) {
                         select(radio);
                    }
                    else {
                        radio.setForeground(fgColor);
                    }
                }
            };
        }
        
        private void add(RadioLabel radio) {
            radioList.add(radio);
            radio.addMouseListener(actionListener);
        }
        
        private void setSelection(RadioLabel radio) {
            for (RadioLabel rl : radioList) {
                rl.setSelected(false);
                rl.setForeground(fgColor);
            }
            
            radio.setSelected(true);
            radio.setForeground(selectColor);
        }
        
        private void select(RadioLabel radio) {
            setSelection(radio);
            selectionChanged(radio);
        }
        
        void selectionChanged(RadioLabel radio) {
        }
    }   
}
